/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.diagram.merge.ui.command.handler;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Iterator;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.expressions.IEvaluationContext;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.compare.diff.metamodel.ComparisonResourceSnapshot;
import org.eclipse.emf.compare.match.metamodel.MatchModel;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.ISources;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.uml2.uml.resource.UMLResource;
import org.modelversioning.conflictreport.ConflictReport;
import org.modelversioning.conflicts.detection.IThreeWayDiffProvider;
import org.modelversioning.conflicts.detection.impl.ConflictDetectionEngineImpl;
import org.modelversioning.conflicts.detection.impl.ThreeWayDiffProvider;
import org.modelversioning.conflicts.detection.impl.UMLAddAddConflictDetector;
import org.modelversioning.conflicts.detection.impl.UMLDeleteUseConflictDetector;
import org.modelversioning.core.diff.service.DiffService;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;
import org.modelversioning.core.match.MatchException;
import org.modelversioning.core.match.service.MatchService;
import org.modelversioning.core.util.ModelDiagramUtil;
import org.modelversioning.diagram.merge.ui.Activator;
import org.modelversioning.merge.conflictdiagram.ConflictDiagramPlugin;
import org.modelversioning.merge.conflictdiagram.engine.IConflictDiagramMergeEngine;
import org.modelversioning.merge.conflictdiagram.engine.impl.ConflictDiagramMergeEngine;
import org.modelversioning.operations.detection.service.OperationDetectionService;

/**
 * This class realizes the {@link AbstractHandler} for the merge command.
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a> *
 */
public class MergeCommandHandler extends AbstractHandler {

	private boolean enabled = false;

	private ISelection selection;
	protected ITreeSelection treeSelection;
	private ResourceSet resourceSet = new ResourceSetImpl();
	private IFile[] selectedFiles = new IFile[3];
	private Shell shell;

	public static final String WORKING_COPY_1 = "working_copy_1";
	public static final String WORKING_COPY_2 = "working_copy_2";

	private URI mergedResourceURI;

	@Override
	public boolean isEnabled() {
		return enabled;
	}

	@Override
	public void setEnabled(Object evaluationContext) {
		enabled = false;

		if (evaluationContext != null
				&& evaluationContext instanceof IEvaluationContext) {
			IEvaluationContext context = (IEvaluationContext) evaluationContext;

			Object currentSelection = context
					.getVariable(ISources.ACTIVE_CURRENT_SELECTION_NAME);

			if (currentSelection != null
					&& currentSelection instanceof ISelection) {
				selection = (ISelection) currentSelection;

				if (selection instanceof ITreeSelection) {
					treeSelection = (ITreeSelection) selection;

					if (treeSelection.size() == 3) {
						int i = 0;
						Iterator<IStructuredSelection> iterator = treeSelection
								.iterator();

						while (iterator.hasNext()) {
							selectedFiles[i] = (IFile) Platform
									.getAdapterManager().getAdapter(
											iterator.next(), IFile.class);
							i++;
						}
						if (selectedFiles[0].getFileExtension().equals(
								selectedFiles[1].getFileExtension())
								&& selectedFiles[0].getFileExtension().equals(
										selectedFiles[2].getFileExtension())) {
							enabled = true;
						}
					}
				}
			}
		}
	}

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		Assert.isTrue(selection.equals(HandlerUtil.getCurrentSelection(event)),
				"the selection is not equal.");

		shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
		ElementListSelectionDialog dialog = new ElementListSelectionDialog(
				shell, new WorkbenchLabelProvider());
		dialog.setAllowDuplicates(false);
		dialog.setElements(selectedFiles);
		dialog.setMultipleSelection(false);
		dialog.setMessage("Select original model version");
		dialog.setTitle("Select original model");
		if (dialog.open() == ElementListSelectionDialog.OK) {

			// determine files
			IFile originFile = (IFile) dialog.getFirstResult();
			IFile otherFile1 = null;
			IFile otherFile2 = null;
			if (originFile.equals(selectedFiles[0])) {
				otherFile1 = selectedFiles[1];
				otherFile2 = selectedFiles[2];
			}
			if (originFile.equals(selectedFiles[1])) {
				otherFile1 = selectedFiles[0];
				otherFile2 = selectedFiles[2];
			}
			if (originFile.equals(selectedFiles[2])) {
				otherFile1 = selectedFiles[0];
				otherFile2 = selectedFiles[1];
			}

			try {
				process(originFile, otherFile1, otherFile2);
			} catch (RuntimeException e) {
				reportException(e);
			} catch (InvocationTargetException e) {
				reportException(e);
			} catch (InterruptedException e) {
				reportException(e);
			}
		}

		return null;
	}

	/**
	 * Generates the conflict diagram
	 * 
	 * @param originFile
	 * @param otherFile1
	 * @param otherFile2
	 * @throws InvocationTargetException
	 * @throws InterruptedException
	 */
	private void process(final IFile originFile, final IFile otherFile1,
			final IFile otherFile2) throws InvocationTargetException,
			InterruptedException {
		Activator.getDefault().getWorkbench().getProgressService()
				.run(false, false, new IRunnableWithProgress() {
					@Override
					public void run(IProgressMonitor progressMonitor)
							throws InvocationTargetException,
							InterruptedException {

						SubMonitor monitor = SubMonitor
								.convert(progressMonitor);

						monitor.beginTask("Creating Conflict Diagram", 9);

						try {
							// reading files
							monitor.subTask("Reading files");
							initResourceSet();

							boolean diagramMerge = false;

							Resource originDiagramResource = null;
							Resource originModelResource = null;
							Resource otherDiagramResource1 = null;
							Resource otherModelResource1 = null;
							Resource otherDiagramResource2 = null;
							Resource otherModelResource2 = null;

							// merging diagrams?
							if (ModelDiagramUtil.containsDiagram(resourceSet.getResource(
									URI.createURI(originFile.getLocationURI()
											.toString()), true))) {
								diagramMerge = true;

								originDiagramResource = resourceSet
										.getResource(URI.createURI(originFile
												.getLocationURI().toString()),
												true);

								IFile originModelFile = ModelDiagramUtil
										.getModelFilesFromDiagram(
												originDiagramResource)
										.toArray(
												new IFile[ModelDiagramUtil
														.getModelFilesFromDiagram(
																originDiagramResource)
														.size()])[0];
								originModelResource = resourceSet.getResource(
										URI.createURI(originModelFile
												.getLocationURI().toString(),
												true), true);

								otherDiagramResource1 = resourceSet
										.getResource(URI.createURI(otherFile1
												.getLocationURI().toString(),
												true), true);
								IFile otherModelFile1 = ModelDiagramUtil
										.getModelFilesFromDiagram(
												otherDiagramResource1)
										.toArray(
												new IFile[ModelDiagramUtil
														.getModelFilesFromDiagram(
																otherDiagramResource1)
														.size()])[0];
								otherModelResource1 = resourceSet.getResource(
										URI.createURI(otherModelFile1
												.getLocationURI().toString(),
												true), true);

								otherDiagramResource2 = resourceSet
										.getResource(URI.createURI(otherFile2
												.getLocationURI().toString(),
												true), true);
								IFile otherModelFile2 = ModelDiagramUtil
										.getModelFilesFromDiagram(
												otherDiagramResource2)
										.toArray(
												new IFile[ModelDiagramUtil
														.getModelFilesFromDiagram(
																otherDiagramResource2)
														.size()])[0];
								otherModelResource2 = resourceSet.getResource(
										URI.createURI(otherModelFile2
												.getLocationURI().toString(),
												true), true);
							} else {
								originModelResource = resourceSet.getResource(
										URI.createURI(originFile
												.getLocationURI().toString(),
												true), true);
								otherModelResource1 = resourceSet.getResource(
										URI.createURI(otherFile1
												.getLocationURI().toString(),
												true), true);
								otherModelResource2 = resourceSet.getResource(
										URI.createURI(otherFile2
												.getLocationURI().toString(),
												true), true);
							}

							EList<Resource> mergedResources = new BasicEList<Resource>();
							Resource mergedModelResource = resourceSet.createResource(URI
									.createURI(originModelResource.getURI()
											.trimFileExtension()
											+ ConflictDiagramPlugin.MERGED_MODEL_POSTFIX
											+ "."
											+ originModelResource.getURI()
													.fileExtension()));
							mergedResources.add(mergedModelResource);
							mergedResourceURI = mergedModelResource.getURI();
							if (diagramMerge) {
								Resource mergedDiagramResource = resourceSet.createResource(URI
										.createURI(originDiagramResource
												.getURI().trimFileExtension()
												+ ConflictDiagramPlugin.MERGED_MODEL_POSTFIX
												+ "."
												+ originDiagramResource
														.getURI()
														.fileExtension()));
								mergedResources.add(mergedDiagramResource);
								mergedResourceURI = mergedDiagramResource
										.getURI();
							}
							monitor.worked(1);

							// Matching
							monitor.subTask("Matching models");
							MatchService matchService = new MatchService();
							MatchModel modelMatch1 = matchService
									.generateMatchModel(originModelResource,
											otherModelResource1);
							MatchModel modelMatch2 = matchService
									.generateMatchModel(originModelResource,
											otherModelResource2);

							MatchModel diagramMatch1 = null;
							MatchModel diagramMatch2 = null;

							if (diagramMerge) {
								diagramMatch1 = matchService
										.generateMatchModel(
												originDiagramResource,
												otherDiagramResource1);
								diagramMatch2 = matchService
										.generateMatchModel(
												originDiagramResource,
												otherDiagramResource2);
							}
							monitor.worked(1);

							// Diffing
							monitor.subTask("Diffing models");
							DiffService diffService = new org.modelversioning.core.diff.service.DiffService();
							ComparisonResourceSnapshot modelCrs1 = diffService
									.generateComparisonResourceSnapshot(modelMatch1);
							ComparisonResourceSnapshot modelCrs2 = diffService
									.generateComparisonResourceSnapshot(modelMatch2);
							OperationDetectionService operationDetectionService = new OperationDetectionService();
							operationDetectionService
									.findAndAddOperationOccurrences(modelCrs1);
							operationDetectionService
									.findAndAddOperationOccurrences(modelCrs2);

							ComparisonResourceSnapshot diagramCrs1 = null;
							ComparisonResourceSnapshot diagramCrs2 = null;

							if (diagramMerge) {
								diagramCrs1 = diffService
										.generateComparisonResourceSnapshot(diagramMatch1);
								diagramCrs2 = diffService
										.generateComparisonResourceSnapshot(diagramMatch2);
							}
							monitor.worked(1);

							// Creating DiffProvider
							monitor.subTask("Reading diffs");
							IThreeWayDiffProvider modelThreeWayDiff = new ThreeWayDiffProvider(
									modelCrs1, modelCrs2);
							monitor.worked(1);

							// Detect Conflicts
							monitor.subTask("Detecting conflicts");
							ConflictDetectionEngineImpl engine = new ConflictDetectionEngineImpl();
							engine.getOverlappingChangeDetectors().add(
									new UMLAddAddConflictDetector());
//							engine.getOverlappingChangeDetectors().add(
//									new UMLDeleteUseConflictDetector());

							engine.setOperationContractViolationAware(true);
							ConflictReport conflictReport = engine
									.detectConflicts(modelThreeWayDiff,
											new SubProgressMonitor(monitor, 1));

							if (diagramMerge) {
								conflictReport.getLeftDiagrams().add(
										diagramCrs1);
								conflictReport.getRightDiagrams().add(
										diagramCrs2);
							}
							conflictReport.setLeftUser("Harry");
							conflictReport.setRightUser("Sally");
							monitor.worked(1);

							// Save Conflict Report
							monitor.subTask("Saving conflict report");
							IFile conflictReportFile = originFile
									.getParent()
									.getFile(
											new Path(
													originFile.getName()
															+ "." //$NON-NLS-1$
															+ ConflictDiagramPlugin.CONFLICT_REPORT_EXTENSION));
							if (conflictReportFile != null) {
								Resource conflictReportResource = resourceSet
										.createResource(URI
												.createPlatformResourceURI(
														conflictReportFile
																.getFullPath()
																.toString(),
														true));
								conflictReportResource.getContents().add(
										conflictReport);
								conflictReportResource.save(Collections
										.emptyMap());
							}
							// }
							monitor.worked(1);

							// Generate Conflict Diagram
							monitor.subTask("Generating conflict diagram");
							IConflictDiagramMergeEngine merger = new ConflictDiagramMergeEngine();
							merger.merge(conflictReport, mergedResources,
									monitor);
							monitor.worked(1);

							// Saving Conflict Diagram
							monitor.subTask("Saving conflict diagram");
							for (Resource r : mergedResources) {
								r.save(Collections.EMPTY_MAP);
							}
							monitor.worked(1);

							// Open merged result
							monitor.subTask("Loading conflict diagram");
							IWorkbench wb = PlatformUI.getWorkbench();
							IWorkbenchPage page = wb.getActiveWorkbenchWindow()
									.getActivePage();

							IFile mergedFile = originFile.getParent().getFile(
									new Path(mergedResourceURI.lastSegment()));

							System.out.println(mergedFile);

							IEditorDescriptor desc = PlatformUI.getWorkbench()
									.getEditorRegistry()
									.getDefaultEditor(mergedFile.getName());
							if (desc != null) {
								page.openEditor(
										new FileEditorInput(mergedFile),
										desc.getId());
							}
							else {
								page.openEditor(new FileEditorInput(mergedFile), "org.eclipse.emf.ecore.presentation.EcoreEditorID");
							}
							monitor.worked(1);

						} catch (IOException e) {
							reportException(e);
						} catch (MatchException e) {
							reportException(e);
						} catch (PartInitException e) {
							reportException(e);
						} finally {
							monitor.done();
						}
					}
				});
	}

	/**
	 * Initializes a {@link ResourceSet}.
	 */
	private void initResourceSet() {
		// load resources
		resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(UMLResource.FILE_EXTENSION, UMLResource.Factory.INSTANCE);
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put("operation", new EcoreResourceFactoryImpl());
		resourceSet
				.getResourceFactoryRegistry()
				.getExtensionToFactoryMap()
				.put(Resource.Factory.Registry.DEFAULT_EXTENSION,
						new UUIDResourceFactoryImpl());
	}

	/**
	 * @param e
	 */
	private void reportException(Throwable e) {
		IStatus status = new Status(
				IStatus.ERROR,
				Activator.PLUGIN_ID,
				"An error occurred while performing conflict diagram generation",
				e);
		ErrorDialog.openError(shell, "Error occurred", e.getMessage(), status,
				0);
		Activator.getDefault().getLog().log(status);
	}
}