/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.diagram.merge.ui.providers;

import java.util.HashMap;
import java.util.List;

import org.eclipse.draw2d.ConnectionLayer;
import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.IFigure;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.ConnectionEditPart;
import org.eclipse.gmf.runtime.diagram.core.listener.DiagramEventBroker;
import org.eclipse.gmf.runtime.diagram.core.listener.NotificationListener;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.AbstractDecorator;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecorator;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecoratorTarget;
import org.eclipse.gmf.runtime.gef.ui.figures.DefaultSizeNodeFigure;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.swt.graphics.Color;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Stereotype;
import org.modelversioning.conflicts.profile.util.ProfileUtil;

/**
 * A {@link IDecorator} for the Conflict Diagram coloring conflicting elements.
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class ConflictDecorator extends AbstractDecorator {
	/*
	 * The map holding pairs of chainged and origin Figures.
	 */
	private static HashMap<IFigure, IFigure> figureMap;
	/*
	 * The origin figure used to hold origin values.
	 */
	private IFigure originFigure = null;
	/*
	 * The changed figure.
	 */
	private IFigure changedFigure = null;
	/*
	 * The alpha value of the origin Shape.
	 */
	public static final int ORIGINAL_ALPHA = 100;
	/*
	 * The alpha value of conflict decoration.
	 */
	public static final int CONFLICT_ALPHA = 90;
	/*
	 * The foreground color used for conflicts
	 */
	public static final Color CONFLICT_FG = new Color(null, 255, 0, 0);
	/*
	 * The background color used for conflicts
	 */
	public static final Color CONFLICT_BG = new Color(null, 255, 192, 203);

	static {
		figureMap = new HashMap<IFigure, IFigure>();
	}

	public ConflictDecorator(IDecoratorTarget decoratorTarget) {
		super(decoratorTarget);
	}

	static public View getDecoratorTargetNode(IDecoratorTarget decoratorTarget) {
		View node = (View) decoratorTarget.getAdapter(View.class);

		if (node != null && node.getElement() != null) {
			return node;
		}

		return null;
	}

	public void refresh() {
		removeDecoration();

		View node = getDecoratorTargetNode(getDecoratorTarget());
		if (node != null) {
			IGraphicalEditPart editPart = (IGraphicalEditPart) getDecoratorTarget()
					.getAdapter(IGraphicalEditPart.class);

			EObject object = editPart.resolveSemanticElement();

			if (object instanceof Element) {
				Element element = (Element) object;

				if (shouldDecorate(element) && originFigure == null) {

					if (editPart instanceof ConnectionEditPart) {
						changedFigure = editPart.getFigure();
					} else {
						changedFigure = getFigure(editPart.getFigure());
					}

					if (figureMap.get(changedFigure) == null) {
						originFigure = new Figure();
						originFigure.setForegroundColor(new Color(changedFigure
								.getForegroundColor().getDevice(),
								changedFigure.getForegroundColor().getRGB()));
						originFigure.setBackgroundColor(new Color(changedFigure
								.getBackgroundColor().getDevice(),
								changedFigure.getBackgroundColor().getRGB()));
						originFigure.setOpaque(changedFigure.isOpaque() ? true
								: false);

						figureMap.put(changedFigure, originFigure);					

						changedFigure.setForegroundColor(CONFLICT_FG);
						changedFigure.setBackgroundColor(CONFLICT_BG);
						setAlpha(changedFigure, CONFLICT_ALPHA);
						// changedFigure.setOpaque(false);
						
						changedFigure.repaint();
					} else {
						originFigure = figureMap.get(changedFigure);
					}
				}
			}
		}
	}

	private void setAlpha(IFigure figure, int alpha) {
		if (figure instanceof org.eclipse.draw2d.Shape) {
			((org.eclipse.draw2d.Shape) figure).setAlpha(alpha);
			List<Figure> children = figure.getChildren();
			for (Figure child : children) {
				setAlpha(child, 0);
			}
		}
	}

	private boolean shouldDecorate(Element element) {
		for (Stereotype stereotype : element.getAppliedStereotypes()) {
			if (ProfileUtil.getGeneral(stereotype, "Conflict") != null) {
				if (!(Boolean)ProfileUtil.getValue(element, stereotype, ProfileUtil.CONFLICT_IS_RESOLVED_FEATURE)) {
					return true;
				}
			}
		}
		return false;
	}

	private NotificationListener notificationListener = new NotificationListener() {
		public void notifyChanged(Notification notification) {
			refresh();
		}
	};

	public void activate() {
		IGraphicalEditPart editPart = (IGraphicalEditPart) getDecoratorTarget()
				.getAdapter(IGraphicalEditPart.class);
		assert editPart != null;
		DiagramEventBroker.getInstance(editPart.getEditingDomain())
				.addNotificationListener(editPart.resolveSemanticElement(),
						notificationListener);
	}

	public void deactivate() {
		removeDecoration();
		IGraphicalEditPart editPart = (IGraphicalEditPart) getDecoratorTarget()
				.getAdapter(IGraphicalEditPart.class);
		assert editPart != null;
		DiagramEventBroker.getInstance(editPart.getEditingDomain())
				.removeNotificationListener(editPart.resolveSemanticElement(),
						notificationListener);
	}

	@Override
	protected void removeDecoration() {
		super.removeDecoration();

		if (originFigure != null) {
			View node = getDecoratorTargetNode(getDecoratorTarget());
			if (node != null) {
				IGraphicalEditPart editPart = (IGraphicalEditPart) getDecoratorTarget()
						.getAdapter(IGraphicalEditPart.class);

				IFigure figure = null;

				if (editPart instanceof ConnectionEditPart) {
					figure = editPart.getFigure();
				} else {
					figure = getFigure(editPart.getFigure());
				}

				if (figure.equals(changedFigure)) {
					figure.setForegroundColor(originFigure.getForegroundColor());
					figure.setBackgroundColor(originFigure.getBackgroundColor());
					figure.setOpaque(originFigure.isOpaque());
					setAlpha(changedFigure, ORIGINAL_ALPHA);
					
					figure.repaint();

					figureMap.remove(changedFigure);
					originFigure = null;
					changedFigure = null;
				}
			}
		}
	}

	private IFigure getFigure(IFigure figure) {
		if (figure.getParent() == null || figure instanceof ConnectionLayer
				|| figure.getParent() instanceof DefaultSizeNodeFigure) {
			return figure;
		}
		return getFigure(figure.getParent());
	}
}