/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.merge.conflictdiagram.engine.impl;

import java.util.Collection;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.compare.diff.merge.IMergeListener;
import org.eclipse.emf.compare.diff.merge.service.MergeService;
import org.eclipse.emf.compare.diff.metamodel.DiffElement;
import org.eclipse.emf.compare.diff.metamodel.DifferenceKind;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.PlatformUI;
import org.eclipse.uml2.uml.Actor;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLPackage;
import org.modelversioning.conflictreport.ConflictReport;
import org.modelversioning.conflictreport.EquivalentChange;
import org.modelversioning.conflictreport.conflict.Conflict;
import org.modelversioning.conflicts.profile.util.ProfileUtil;
import org.modelversioning.merge.IMergeStrategy;
import org.modelversioning.merge.IMerger;
import org.modelversioning.merge.conflictdiagram.engine.IConflictDiagramMergeEngine;
import org.modelversioning.merge.conflictdiagram.util.DiagramMergeUtil;
import org.modelversioning.merge.conflictdiagram.util.DiagramMergeUtilFactory;
import org.modelversioning.merge.conflictdiagram.util.MergeUtil;
import org.modelversioning.merge.impl.ConflictAwareMergeStrategy;
import org.modelversioning.merge.impl.MergeSpecificChangeStrategy;
import org.modelversioning.merge.impl.MergerImpl;

/**
 * Default implementation of the {@link IConflictDiagramMergeEngine} merging two
 * models and their respective diagrams of a {@link ConflictReport} leaving out
 * conflicting changes and deletions.
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class ConflictDiagramMergeEngine implements IConflictDiagramMergeEngine {

	private IMerger merger = new MergerImpl();

	private IMergeListener mergeListener;

	private IMergeStrategy conflictAwareMergeStrategy = new ConflictAwareMergeStrategy();

	private ConflictReport conflictReport;

	private EList<Resource> mergedResources;

	private MergeUtil mergeUtil;

	private DiagramMergeUtil diagramUtil;

	private DiagramMergeUtilFactory diagramUtilFactory = new DiagramMergeUtilFactory();

	private SubMonitor subMonitor;

	/**
	 * The actual side for merging.
	 */
	private Side side;

	/**
	 * The leftUser.
	 */
	private Actor leftUser;

	/**
	 * The rightUser.
	 */
	private Actor rightUser;

	@Override
	public EList<Resource> merge(ConflictReport conflictReport,
			EList<Resource> mergedResources, IProgressMonitor monitor) {

		this.subMonitor = SubMonitor.convert(monitor);

		this.conflictReport = conflictReport;
		this.mergedResources = mergedResources;

		// check parameters
		Assert.isTrue(conflictReport.getLeftVersion() != null);
		Assert.isTrue(conflictReport.getRightVersion() != null);
		if (conflictReport.getLeftDiagrams() != null
				&& conflictReport.getRightDiagrams() != null) {
			Assert.isTrue(conflictReport.getLeftDiagrams().size() == conflictReport
					.getRightDiagrams().size());
			Assert.isTrue(mergedResources.size() == conflictReport
					.getLeftDiagrams().size() + 1);
		}

		// set strategy for merger
		merger.setMergeStrategy(conflictAwareMergeStrategy);

		// merge model
		mergeModel(true);

		// merge diagram
		mergeDiagrams(true);

		// apply stereotypes
		subMonitor.subTask("Applying Change Stereotypes");
		applyChangeStereotypes(mergedResources.get(0).getContents());
		subMonitor.worked(1);

		// handle conflicts
		subMonitor.subTask("Applying Conflict Stereotypes");
		if (conflictReport.getConflicts() != null) {
			for (Conflict conflict : conflictReport.getConflicts()) {
				mergeUtil.applyConflictStereotype(conflict, diagramUtil);
			}
		}
		if (conflictReport.getEquivalentChanges() != null) {
			for (EquivalentChange equivalentChange : conflictReport
					.getEquivalentChanges()) {
				mergeUtil.applyEquivalentChangeStereotype(equivalentChange);
			}
		}
		subMonitor.worked(1);

		// handle violations
		subMonitor.subTask("Handling Violations");
		// TODO
		subMonitor.worked(1);

		/**
		 * Handle copyReferences BUG: Remove duplicated references in
		 * constrainedElements
		 */
		Collection<Constraint> constraints = EcoreUtil.getObjectsByType(
				mergedResources.get(0).getContents().get(0).eContents(),
				UMLPackage.eINSTANCE.getConstraint());
		for (Constraint c : constraints) {
			Constraint originConstraint = (Constraint) merger
					.getCorrespondingOriginalObject(c);

			if (originConstraint != null) {
				for (Element e : originConstraint.getConstrainedElements()) {
					c.getConstrainedElements().remove(
							merger.getCorrespondingMergedObject(e));
				}
			}
		}

		return mergedResources;
	}

	/**
	 * Merges the conflict's change of the given {@link Side}
	 * 
	 * @param element
	 *            The context element to apply the diff to
	 * @param conflictStereotype
	 *            The conflict stereotype holding the change
	 * @param side
	 *            The side of the change to apply
	 * @param monitor
	 *            The {@link IProgressMonitor} to report to.
	 * @return
	 */
	public EList<Resource> mergeSpecific(Element element,
			Stereotype conflictStereotype, Side side, IProgressMonitor monitor) {
		Assert.isNotNull(conflictReport);
		Assert.isNotNull(mergedResources);

		this.side = side;
		this.subMonitor = SubMonitor.convert(monitor);
		subMonitor.subTask("Preparing Merge");

		Side opposite = Side.RIGHT.equals(side) ? Side.LEFT : Side.RIGHT;

		EList<EObject> applyChangeObjects = new BasicEList<EObject>();
		EList<EObject> revertChangeObjects = new BasicEList<EObject>();
		EList<DiffElement> applyDiffs = new BasicEList<DiffElement>();
		EList<DiffElement> revertDiffs = new BasicEList<DiffElement>();

		MergeSpecificChangeStrategy mergeSpecificStrategy = new MergeSpecificChangeStrategy();
				
		if (side.equals(Side.RIGHT)) {
			applyChangeObjects.addAll((EList<EObject>) ProfileUtil.getValue(
					element, conflictStereotype,
					ProfileUtil.CONFLICT_MY_CHANGE_FEATURE));
			revertChangeObjects.addAll((EList<EObject>) ProfileUtil.getValue(
					element, conflictStereotype,
					ProfileUtil.CONFLICT_THEIR_CHANGE_FEATURE));
		} else {
			applyChangeObjects.addAll((EList<EObject>) ProfileUtil.getValue(
					element, conflictStereotype,
					ProfileUtil.CONFLICT_THEIR_CHANGE_FEATURE));
			revertChangeObjects.addAll((EList<EObject>) ProfileUtil.getValue(
					element, conflictStereotype,
					ProfileUtil.CONFLICT_MY_CHANGE_FEATURE));
		}

		for (EObject applyChangeObject : applyChangeObjects) {
			DiffElement applyDiff = (DiffElement) ProfileUtil.getFeature(
					applyChangeObject, ProfileUtil.CHANGE_DIFF_ELEMENT_FEATURE);

			if (!ProfileUtil.isApplied(applyChangeObject)
					|| !conflictAwareMergeStrategy.shouldMerge(applyDiff,
							conflictReport)
					|| !applyDiff.getKind().equals(DifferenceKind.ADDITION)) {
				applyDiffs.add(applyDiff);
			}

			mergeUtil.updateChangeStereotype(applyDiff, mergeSpecificStrategy,
					conflictReport, side, getActorForSide(side));
			mergeUtil.updateConflictStereotype(applyDiff, conflictStereotype,
					true);
		}
		
		// set diff elements to merge
		mergeSpecificStrategy.setChangesToMerge(applyDiffs);
		
				
		for (EObject revertChangeObject : revertChangeObjects) {
			DiffElement revertDiff = (DiffElement) ProfileUtil
					.getFeature(revertChangeObject,
							ProfileUtil.CHANGE_DIFF_ELEMENT_FEATURE);
			revertDiffs.add(revertDiff);

			mergeUtil.updateChangeStereotype(revertDiff, mergeSpecificStrategy,
					conflictReport, opposite, getActorForSide(opposite));
		}

		if (applyDiffs.size() > 0 && revertDiffs.size() > 0) {
			// EList<DiffElement> diffsToMerge = new BasicEList<DiffElement>();
			// diffsToMerge.addAll(applyDiffs);

			// MergeSpecificChangeStrategy mergeSpecificStrategy = new
			// MergeSpecificChangeStrategy();
			// mergeSpecificStrategy.setChangesToMerge(applyDiffs);
			merger.setMergeStrategy(mergeSpecificStrategy);
			subMonitor.worked(1);

			// apply only changes which are currently not applied
			// if (!ProfileUtil.isApplied(applyChangeObjects)
			// || !conflictAwareMergeStrategy.shouldMerge(applyDiff,
			// conflictReport)) {
			mergeModel(false);
			mergeDiagrams(false);
			// }

			// mergeUtil.updateChangeStereotype(applyDiff,
			// mergeSpecificStrategy,
			// conflictReport, side, getActorForSide(side));
			// mergeUtil.updateChangeStereotype(revertDiff,
			// mergeSpecificStrategy,
			// conflictReport, opposite, getActorForSide(opposite));
			// mergeUtil.updateConflictStereotype(applyDiff, conflictStereotype,
			// true);
		}

		return mergedResources;
	}

	public EList<Resource> revert(Element element, IProgressMonitor monitor) {
		Assert.isNotNull(conflictReport);
		Assert.isNotNull(mergedResources);

		this.subMonitor = SubMonitor.convert(monitor);
		subMonitor.subTask("Preparing Revert");

		// get all changes for the elment
		EList<EObject> changeObjects = mergeUtil.getChanges(element);
		// EList<DiffElement> diffElements = new BasicEList<DiffElement>();

		for (EObject changeObject : changeObjects) {

			DiffElement revertDiff = (DiffElement) ProfileUtil.getFeature(
					changeObject, ProfileUtil.CHANGE_DIFF_ELEMENT_FEATURE);
			if (revertDiff != null) {
				Side side = changeObject.eClass().getName().startsWith("My") ? Side.RIGHT
						: Side.LEFT;

				mergeUtil.updateChangeStereotype(revertDiff,
						new MergeSpecificChangeStrategy(), conflictReport,
						side, getActorForSide(side));
				// diffElements.add(revertDiff);
			}
		}

		// create new diff model for reverting changes
		// DiffModel revertDiff = DiffFactory.eINSTANCE.createDiffModel();
		// DiffGroup diffGroup = DiffFactory.eINSTANCE.createDiffGroup();
		// revertDiff.getOwnedElements().add(diffGroup);
		// diffGroup.getSubDiffElements().addAll(DiffUtil.shallowCopy(diffElements));
		//
		// IMergeListener revertListener = new RevertMergeListener(mergeUtil);
		// MergeService.addMergeListener(revertListener);
		// MergeService.merge(revertDiff.getDifferences(), true);
		// MergeService.removeMergeListener(revertListener);

		return mergedResources;
	}

	/**
	 * Merges the given changes without considering conflicts.
	 * 
	 * @param diffsToMerge
	 *            The actual changes to merge
	 * @param monitor
	 *            The {@link IProgressMonitor} to report to.
	 */
	// public EList<Resource> mergeSpecific(EList<DiffElement> diffsToMerge,
	// IProgressMonitor monitor) {
	// Assert.isNotNull(conflictReport);
	// Assert.isNotNull(mergedResources);
	//
	// this.subMonitor = SubMonitor.convert(monitor);
	//
	// subMonitor.subTask("Preparing Merge");
	// MergeSpecificChangeStrategy mergeSpecificStrategy = new
	// MergeSpecificChangeStrategy();
	// mergeSpecificStrategy.setChangesToMerge(diffsToMerge);
	// merger.setMergeStrategy(mergeSpecificStrategy);
	// subMonitor.worked(1);
	//
	// mergeModel(false);
	//
	// mergeDiagrams(false);
	//
	// for (DiffElement diff : diffsToMerge) {
	// mergeUtil.updateChangeStereotype(diff, mergeSpecificStrategy,
	// conflictReport, Side.RIGHT, getActorForSide(Side.RIGHT));
	// }
	//
	// return mergedResources;
	// }

	/**
	 * Merges the model
	 */
	private void mergeModel(boolean createNewModel) {
		subMonitor.subTask("Merging Models");
		if (createNewModel) {
			merger.merge(conflictReport, mergedResources.get(0),
					subMonitor.newChild(5));

			subMonitor.subTask("Preparing Merge Util");
			// prepare merge util
			mergeUtil = new MergeUtil(mergedResources.get(0));
			subMonitor.worked(1);
		} else {
			((MergerImpl) merger).mergeInplace(conflictReport,
					conflictReport.getLeftVersion(),
					conflictReport.getRightVersion(), mergedResources.get(0),
					subMonitor.newChild(4));
		}
		subMonitor.worked(1);
	}

	/**
	 * Merges the diagrams
	 */
	private void mergeDiagrams(boolean createNewModel) {
		subMonitor.subTask("Merging Diagrams");
		// register mergeListener to get merge events
		mergeListener = new ConflictDiagramMergeListener(mergeUtil);
		MergeService.addMergeListener(mergeListener);

		for (int i = 0; i < conflictReport.getLeftDiagrams().size(); i++) {
			// init diagram merge util
			IEditorDescriptor editorDescriptor = PlatformUI
					.getWorkbench()
					.getEditorRegistry()
					.getDefaultEditor(
							mergedResources.get(i + 1).getURI().path());

			// find diagram merge helper suitable for this diagram type
			diagramUtil = diagramUtilFactory.getDiagramUtil(editorDescriptor);

			// merge diagram
			if (createNewModel) {
				// calculate and set dependent diagram changes
				diagramUtil.annotateDiagramDependencies(conflictReport, i);
				System.out.println("DiagramDependencies: "
						+ conflictReport.getDiagramDependencies());

				merger.merge(conflictReport, conflictReport.getLeftDiagrams()
						.get(i), conflictReport.getRightDiagrams().get(i),
						mergedResources.get(i + 1), subMonitor.newChild(5));

				diagramUtil.initViewElementMap(mergedResources.get(0),
						mergedResources.get(i + 1));
			} else {
				((MergerImpl) merger).mergeInplace(conflictReport,
						conflictReport.getLeftDiagrams().get(i), conflictReport
								.getRightDiagrams().get(i), mergedResources
								.get(i + 1), subMonitor.newChild(4));
			}
		}

		// unregister mergeListener
		MergeService.removeMergeListener(mergeListener);

		subMonitor.worked(1);
	}

	/**
	 * Applies stereotypes for changes
	 * 
	 * @param mergedModels
	 */
	private void applyChangeStereotypes(EList<EObject> mergedModels) {
		// apply profile on the merged model
		for (EObject o : mergedModels) {
			Package package_;
			if (o instanceof Package)
				package_ = (Package) o;
			else {
				package_ = (Package) EcoreUtil.getObjectByType(o.eContents(),
						UMLPackage.Literals.PACKAGE);
				if (package_ == null)
					continue;
			}
			mergeUtil.ensureProfileApplication(package_);
		}

		// find or add Actors for left and right user
		if (mergedModels.get(0) instanceof Package) {
			Package package_ = (Package) mergedModels.get(0);

			leftUser = (Actor) package_
					.getPackagedElement(getUserForSide(Side.LEFT));
			if (leftUser == null)
				leftUser = (Actor) package_.createPackagedElement(
						getUserForSide(Side.LEFT), UMLPackage.Literals.ACTOR);

			rightUser = (Actor) package_
					.getPackagedElement(getUserForSide(Side.RIGHT));
			if (rightUser == null)
				rightUser = (Actor) package_.createPackagedElement(
						getUserForSide(Side.RIGHT), UMLPackage.Literals.ACTOR);
		}

		for (DiffElement diff : conflictReport.getLeftVersion().getDiff()
				.getDifferences()) {
			mergeUtil.applyChangeStereotype(diff, conflictAwareMergeStrategy,
					conflictReport, Side.LEFT, leftUser, diagramUtil);
		}

		for (DiffElement diff : conflictReport.getRightVersion().getDiff()
				.getDifferences()) {
			mergeUtil.applyChangeStereotype(diff, conflictAwareMergeStrategy,
					conflictReport, Side.RIGHT, rightUser, diagramUtil);
		}
	}

	private String getUserForSide(Side side) {
		if (side.equals(Side.LEFT))
			return this.conflictReport.getLeftUser();
		else if (side.equals(Side.RIGHT))
			return this.conflictReport.getRightUser();
		return null;
	}

	private Actor getActorForSide(Side side) {
		if (side.equals(Side.LEFT))
			return this.leftUser;
		else if (side.equals(Side.RIGHT))
			return this.rightUser;
		return null;
	}
}
