/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.merge.conflictdiagram.util;

import java.awt.geom.AffineTransform;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.gmf.runtime.diagram.core.services.ViewService;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.notation.Bounds;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.NotationFactory;
import org.eclipse.gmf.runtime.notation.Shape;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.uml2.diagram.clazz.part.UMLDiagramEditorPlugin;
import org.eclipse.uml2.diagram.clazz.part.UMLLinkDescriptor;
import org.eclipse.uml2.diagram.clazz.part.UMLVisualIDRegistry;
import org.eclipse.uml2.diagram.clazz.providers.UMLElementTypes;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;

/**
 * Specific {@link DiagramMergeUtil} for the UML2 Class Diagram Editor with
 * editor ID <code>org.eclipse.uml2.diagram.clazz.part.UMLDiagramEditorID</code>
 * .
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class ClassDiagramMergeUtil extends DiagramMergeUtil {

	public ClassDiagramMergeUtil() {
		super();
	}

	@Override
	public void createShapes(NamedElement leftTarget, NamedElement rightTarget,
			Collaboration conflictElement, EList<Dependency> leftDependencies,
			EList<Dependency> rightDependencies) {

		// get view element representing left target element
		View leftTargetView = getCorrespondingView(leftTarget).get(0);

		// get view element representing right target element
		View rightTargetView = getCorrespondingView(rightTarget).get(0);

		// no view elements found? then no collaboration view element should be
		// created
		if (leftTargetView == null && rightTargetView == null) {
			return;
		}

		// get container view element
		View containerView = getCorrespondingView(leftTarget.eContainer()).get(
				0);

		// container is null
		if (containerView == null) {
			return;
		}

		// editor cannot create collaboration node
		if (!UMLVisualIDRegistry.canCreateNode(containerView,
				UMLVisualIDRegistry.getNodeVisualID(containerView,
						conflictElement))) {
			return;
		}

		// create collaboration view element
		Node conflictElementView = ViewService.createNode(containerView,
				conflictElement, UMLVisualIDRegistry
						.getType(UMLVisualIDRegistry.getNodeVisualID(
								containerView, conflictElement)),
				UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);

		// find suitable location for the newly created collaboration view
		// element
		Bounds bounds = getCenterBounds(leftTargetView, rightTargetView,
				(Shape) conflictElementView);
		if (bounds != null) {
			conflictElementView.setLayoutConstraint(bounds);
		}

		// create left dependency links
		for (Dependency leftDependency : leftDependencies) {
			UMLLinkDescriptor linkDescLeftDependency = new UMLLinkDescriptor(
					conflictElement, leftDependency.getTargets().get(0), // leftTarget,
					UMLElementTypes.getElementType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(leftDependency)),
					UMLVisualIDRegistry
							.getLinkWithClassVisualID(leftDependency));

			Edge leftDependencyView = ViewService.getInstance().createEdge(
					linkDescLeftDependency.getSemanticAdapter(),
					containerView,
					UMLVisualIDRegistry.getType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(leftDependency)),
					ViewUtil.APPEND, true,
					UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
			leftDependencyView.setElement(leftDependency);
			leftDependencyView.setSource(conflictElementView);
			leftTargetView = getCorrespondingView(
					leftDependency.getTargets().get(0)).get(0);
			leftDependencyView.setTarget(leftTargetView);
		}

		// create right dependency links
		for (Dependency rightDependency : rightDependencies) {
			UMLLinkDescriptor linkDescRightDependency = new UMLLinkDescriptor(
					conflictElement, rightDependency.getTargets().get(0), // rightTarget,
					UMLElementTypes.getElementType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(rightDependency)),
					UMLVisualIDRegistry
							.getLinkWithClassVisualID(rightDependency));

			Edge rightDependencyView = ViewService.getInstance().createEdge(
					linkDescRightDependency.getSemanticAdapter(),
					containerView,
					UMLVisualIDRegistry.getType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(rightDependency)),
					ViewUtil.APPEND, true,
					UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
			rightDependencyView.setElement(rightDependency);
			rightDependencyView.setSource(conflictElementView);
			rightTargetView = getCorrespondingView(
					rightDependency.getTargets().get(0)).get(0);
			rightDependencyView.setTarget(rightTargetView);
		}
	}

	@Override
	public void createShapes(Collaboration conflictElement) {

		EList<Dependency> dependencies = conflictElement
				.getClientDependencies();
		EList<Element> clients = new BasicEList<Element>();
		EList<View> clientViews = new BasicEList<View>();
		Element containerElement = null;

		for (Dependency dependency : dependencies) {
			for (Element target : dependency.getTargets()) {
				clients.add(target);
				clientViews.add(getCorrespondingView(target).get(0));

				if (containerElement == null) {
					containerElement = target.getOwner();
				} else {
					containerElement = MergeUtil.getNearestCommonContainer(
							containerElement, target, Element.class);
				}
			}
		}

		// get container view element
		View containerView = getCorrespondingView(containerElement).get(0);

		// container is null
		if (containerView == null) {
			return;
		}

		// editor cannot create collaboration node
		if (!UMLVisualIDRegistry.canCreateNode(containerView,
				UMLVisualIDRegistry.getNodeVisualID(containerView,
						conflictElement))) {
			return;
		}

		// create collaboration view element
		Node conflictElementView = ViewService.createNode(containerView,
				conflictElement, UMLVisualIDRegistry
						.getType(UMLVisualIDRegistry.getNodeVisualID(
								containerView, conflictElement)),
				UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
		
		// add created Edge to View Mapping Table
		EList<View> collaborationViewList = new BasicEList<View>();
		collaborationViewList.add(conflictElementView);
		viewElementMap.put(conflictElement, collaborationViewList);

		// find suitable location for the newly created collaboration view
		// element
		Bounds bounds = getCenterBounds(clientViews,
				(Shape) conflictElementView);
		if (bounds != null) {
			conflictElementView.setLayoutConstraint(bounds);
		}

		// create left dependency links
		for (Dependency dependency : dependencies) {
			UMLLinkDescriptor linkDesc = new UMLLinkDescriptor(
					conflictElement, dependency.getTargets().get(0),
					UMLElementTypes.getElementType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(dependency)),
					UMLVisualIDRegistry
							.getLinkWithClassVisualID(dependency));

			Edge dependencyView = ViewService.getInstance().createEdge(
					linkDesc.getSemanticAdapter(),
					containerView,
					UMLVisualIDRegistry.getType(UMLVisualIDRegistry
							.getLinkWithClassVisualID(dependency)),
					ViewUtil.APPEND, true,
					UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
			
			dependencyView.setSource(conflictElementView);			
			dependencyView.setTarget(getTargetShape(dependency.getTargets().get(0)));
			dependencyView.setElement(dependency);
			
			// add created Edge to View Mapping Table
			EList<View> dependencyViewList = new BasicEList<View>();
			dependencyViewList.add(dependencyView);
			viewElementMap.put(dependency, dependencyViewList);
		}
	}
	
	@Override
	public void updateShapes(Collaboration conflictElement) {

		for (Dependency dependency : conflictElement.getClientDependencies()) {
			if (dependency.getAppliedStereotypes().isEmpty()) {
				View dependencyView = getCorrespondingView(dependency).get(0);
				dependencyView.setVisible(false);
			}
		}
	}

	private View getTargetShape(Element element) {
		View view = getCorrespondingView(element).get(0);
		
//		while ((view == null || !(view instanceof Shape)) && element.getOwner() != null) {
//			view = getTargetShape(element.getOwner());
//		}
		
		return view;
	}

	protected Bounds getCenterBounds(View leftTargetView, View rightTargetView,
			Shape conflictElementView) {

		Bounds bounds = null;
		Bounds bounds1 = null;
		Bounds bounds2 = null;

		// analyze leftTargetView
		if (leftTargetView instanceof Shape) {
			bounds1 = (Bounds) ((Shape) leftTargetView).getLayoutConstraint();
		} else if (leftTargetView instanceof Edge) {
			View leftSource = ((Edge) leftTargetView).getSource();
			View leftTarget = ((Edge) leftTargetView).getTarget();
			if (leftSource instanceof Shape) {
				Bounds srcBounds = (Bounds) ((Shape) leftSource)
						.getLayoutConstraint();
				Bounds targetBounds = (Bounds) ((Shape) leftTarget)
						.getLayoutConstraint();

				org.eclipse.draw2d.geometry.PointList points = new org.eclipse.draw2d.geometry.PointList();
				points.addPoint(srcBounds.getX(), srcBounds.getY());
				points.addPoint(targetBounds.getX(), targetBounds.getY());

				org.eclipse.draw2d.geometry.Point midpoint = points
						.getMidpoint();
				bounds1 = NotationFactory.eINSTANCE.createBounds();
				bounds1.setX(midpoint.x);
				bounds1.setY(midpoint.y);
			}
		}

		// analyze rightTargetView
		if (rightTargetView instanceof Shape) {
			Bounds targetBounds = bounds2 = (Bounds) ((Shape) rightTargetView)
					.getLayoutConstraint();
			org.eclipse.draw2d.geometry.PointList points = new org.eclipse.draw2d.geometry.PointList();
			points.addPoint(bounds1.getX(), bounds1.getY());
			points.addPoint(targetBounds.getX(), targetBounds.getY());

			Point midpoint = points.getMidpoint();
			bounds2 = NotationFactory.eINSTANCE.createBounds();
			bounds2.setX(midpoint.x);
			bounds2.setY(midpoint.y);

		} else if (rightTargetView instanceof Edge) {
			View rightSource = ((Edge) rightTargetView).getSource();
			View rightTarget = ((Edge) rightTargetView).getTarget();
			if (rightSource instanceof Shape) {
				Bounds srcBounds = (Bounds) ((Shape) rightSource)
						.getLayoutConstraint();
				Bounds targetBounds = (Bounds) ((Shape) rightTarget)
						.getLayoutConstraint();

				org.eclipse.draw2d.geometry.PointList points = new org.eclipse.draw2d.geometry.PointList();
				points.addPoint(srcBounds.getX(), srcBounds.getY());
				points.addPoint(targetBounds.getX(), targetBounds.getY());

				Point midpoint = points.getMidpoint();
				bounds2 = NotationFactory.eINSTANCE.createBounds();
				bounds2.setX(midpoint.x);
				bounds2.setY(midpoint.y);
			}
		}

		if (bounds1 != null && bounds2 != null) {
			bounds = (Bounds) conflictElementView
					.createLayoutConstraint(NotationFactory.eINSTANCE
							.createBounds().eClass());

			Point leftPoint = new Point();
			Point rightPoint = new Point();

			if (bounds1.getX() > bounds2.getX()) {
				rightPoint.setLocation(bounds1.getX(), bounds1.getY());
				leftPoint.setLocation(bounds2.getX(), bounds2.getY());
			} else {
				leftPoint.setLocation(bounds1.getX(), bounds1.getY());
				rightPoint.setLocation(bounds2.getX(), bounds2.getY());
			}

			AffineTransform at = AffineTransform.getRotateInstance(
					Math.toRadians(-60), rightPoint.x, rightPoint.y);
			// Math.toRadians(60), leftPoint.x, leftPoint.y);

			at.translate(leftPoint.x, leftPoint.y);
			// at.translate(rightPoint.x, rightPoint.y);

			double translate_x = at.getTranslateX();
			double translate_y = at.getTranslateY();

			bounds.setX((int) Math.round(translate_x));
			bounds.setY((int) Math.round(translate_y));

			if ((bounds2.getX() - bounds1.getX()) < 100) {
				int x = (int) Math.round(translate_x) - 100;
				if (x > 0) {
					bounds.setX(x);
				} else {
					bounds.setX(0);
				}
			}

			// Diagram diagram = conflictElementView.getDiagram();
			// EList<View> viewElements = diagram.getChildren();
			// for (View viewElement : viewElements) {
			// if (viewElement instanceof Shape) {
			// Shape shape = (Shape) viewElement;
			// Bounds b = (Bounds) shape.getLayoutConstraint();
			// // TODO: calculate bounding box and find white space for the new
			// shape
			// }
			// }
		}
		return bounds;
	}

	protected Bounds getCenterBounds(EList<View> contextViews,
			Shape conflictElementView) {

		Bounds bounds = null;
		Bounds leftmostBounds = null;
		Bounds rightmostBounds = null;
		Bounds centerBounds = null;

		// analyze contextView Elements
		for (View view : contextViews) {
			if (view instanceof Shape) {
				bounds = (Bounds) ((Shape) view).getLayoutConstraint();
			} else if (view instanceof Edge) {
				View leftSource = ((Edge) view).getSource();
				View leftTarget = ((Edge) view).getTarget();
				if (leftSource instanceof Shape) {
					Bounds srcBounds = (Bounds) ((Shape) leftSource)
							.getLayoutConstraint();
					Bounds targetBounds = (Bounds) ((Shape) leftTarget)
							.getLayoutConstraint();

					org.eclipse.draw2d.geometry.PointList points = new org.eclipse.draw2d.geometry.PointList();
					points.addPoint(srcBounds.getX(), srcBounds.getY());
					points.addPoint(targetBounds.getX(), targetBounds.getY());

					org.eclipse.draw2d.geometry.Point midpoint = points
							.getMidpoint();
					bounds = NotationFactory.eINSTANCE.createBounds();
					bounds.setX(midpoint.x);
					bounds.setY(midpoint.y);
				}
			}
			if (leftmostBounds == null || leftmostBounds.getX() > bounds.getX()) {
				leftmostBounds = bounds;
			}

			if (rightmostBounds == null
					|| rightmostBounds.getX() < bounds.getX()) {
				rightmostBounds = bounds;
			}
		}

		if (leftmostBounds != null && rightmostBounds != null) {
			centerBounds = (Bounds) conflictElementView
					.createLayoutConstraint(NotationFactory.eINSTANCE
							.createBounds().eClass());

			Point leftPoint = new Point();
			Point rightPoint = new Point();

			leftPoint.setLocation(leftmostBounds.getX(), leftmostBounds.getY());
			rightPoint.setLocation(rightmostBounds.getX(),
					rightmostBounds.getY());

			AffineTransform at = AffineTransform.getRotateInstance(
					Math.toRadians(-60), rightPoint.x, rightPoint.y);
			at.translate(leftPoint.x, leftPoint.y);

			double translate_x = at.getTranslateX();
			double translate_y = at.getTranslateY();

			centerBounds.setX((int) Math.round(translate_x));
			centerBounds.setY((int) Math.round(translate_y));

			if ((rightmostBounds.getX() - leftmostBounds.getX()) < 100) {
				int x = (int) Math.round(translate_x) - 100;
				if (x > 0) {
					centerBounds.setX(x);
				} else {
					centerBounds.setX(0);
				}
			}
		}
		return centerBounds;
	}
}