/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.merge.conflictdiagram.util;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.compare.diff.metamodel.ComparisonResourceSnapshot;
import org.eclipse.emf.compare.diff.metamodel.DiffElement;
import org.eclipse.emf.compare.diff.metamodel.DifferenceKind;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.NamedElement;
import org.modelversioning.conflictreport.ConflictReport;
import org.modelversioning.conflictreport.ConflictReportFactory;
import org.modelversioning.conflictreport.DependentDiagramChange;
import org.modelversioning.core.diff.util.DiffUtil;

/**
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public abstract class DiagramMergeUtil {

	protected Map<EObject, EList<View>> viewElementMap;

	protected Resource mergedModelResource;

	protected Resource mergedDiagramResource;

	public DiagramMergeUtil(/*
							 * Resource mergedModelResource, Resource
							 * mergedDiagramResource
							 */) {
		// this.mergedModelResource = mergedModelResource;
		// this.mergedDiagramResource = mergedDiagramResource;

		// viewElementMap = new HashMap<EObject, EList<View>>();
		//
		// EList<EObject> diagramRoots = new BasicEList<EObject>();
		// diagramRoots.addAll(mergedDiagramResource.getContents());
		//
		// // map diagram elements to model elements
		// for (EObject diagramElement : diagramRoots) {
		// mapViewElement(diagramElement);
		//
		// TreeIterator<EObject> diagramIterator =
		// diagramElement.eAllContents();
		// while(diagramIterator.hasNext()) {
		// mapViewElement(diagramIterator.next());
		// }
		// }
	}

	public void initViewElementMap(Resource mergedModelResource,
			Resource mergedDiagramResource) {
		// run only once per instance
		if (viewElementMap == null) {

			this.mergedModelResource = mergedModelResource;
			this.mergedDiagramResource = mergedDiagramResource;

			viewElementMap = new HashMap<EObject, EList<View>>();

			EList<EObject> diagramRoots = new BasicEList<EObject>();
			diagramRoots.addAll(mergedDiagramResource.getContents());

			// map diagram elements to model elements
			for (EObject diagramElement : diagramRoots) {
				mapViewElement(diagramElement);

				TreeIterator<EObject> diagramIterator = diagramElement
						.eAllContents();
				while (diagramIterator.hasNext()) {
					mapViewElement(diagramIterator.next());
				}
			}
		}
	}

	private void mapViewElement(EObject diagramElement) {
		if (diagramElement instanceof View) {
			View viewElement = (View) diagramElement;
			EObject modelElement = viewElement.getElement();
			if (modelElement != null) {
				if (viewElementMap.containsKey(modelElement)) {
					viewElementMap.get(modelElement).add(viewElement);
				} else {
					EList<View> viewElementList = new BasicEList<View>();
					viewElementList.add(viewElement);
					viewElementMap.put(modelElement, viewElementList);
				}
			}
		}
	}

	/**
	 * Returns the corresponding view element of a model element.
	 * 
	 * @param element
	 *            The model element.
	 * @return The corresponding view element.
	 */
	public EList<View> getCorrespondingView(EObject element) {
		return viewElementMap.get(element);
	}

	/**
	 * Creates shapes for Collaborations and dependencies. This method is
	 * intended to be overwritten by editor-specific subclasses.
	 * 
	 * @param leftTarget
	 * @param rightTarget
	 * @param conflictElement
	 * @param leftDependencies
	 * @param rightDependencies
	 */
	public abstract void createShapes(NamedElement leftTarget,
			NamedElement rightTarget, Collaboration conflictElement,
			EList<Dependency> leftDependencies,
			EList<Dependency> rightDependencies);
	
	/**
	 * Creates shapes for Collaborations and dependencies. This method is
	 * intended to be overwritten by editor-specific subclasses.
	 * 
	 * @param conflictElement
	 */
	public abstract void createShapes(Collaboration conflictElement);
	
	/**
	 * Updates shapes for Collaborations and dependencies. This method is
	 * intended to be overwritten by editor-specific subclasses.
	 * 
	 * @param conflictElement
	 */
	public abstract void updateShapes(Collaboration conflictElement);

	/**
	 * Adds {@link DependentDiagramChange} elements to the conflictReport.
	 * 
	 * @param conflictReport
	 * @param diagramPosition
	 */
	public void annotateDiagramDependencies(ConflictReport conflictReport,
			int diagramPosition) {
		// reset and calculate diagram dependencies
		conflictReport.getDiagramDependencies().clear();

		List<DiffElement> leftModelDifferences = conflictReport
				.getLeftVersion().getDiff().getDifferences();
		List<DiffElement> rightModelDifferences = conflictReport
				.getRightVersion().getDiff().getDifferences();

		for (DiffElement leftModelDiff : leftModelDifferences) {
			for (DiffElement diff : DiffUtil
					.getEffectiveDiffElements(leftModelDiff)) {
				DependentDiagramChange dependentDiagramChange = findDependencies(
						diff,
						conflictReport.getLeftDiagrams().get(diagramPosition));
				if (dependentDiagramChange != null) {
					conflictReport.getDiagramDependencies().add(
							dependentDiagramChange);
				}
			}
		}

		for (DiffElement rightModelDiff : rightModelDifferences) {
			for (DiffElement diff : DiffUtil
					.getEffectiveDiffElements(rightModelDiff)) {
				DependentDiagramChange dependentDiagramChange = findDependencies(
						diff,
						conflictReport.getRightDiagrams().get(diagramPosition));
				if (dependentDiagramChange != null) {
					conflictReport.getDiagramDependencies().add(
							dependentDiagramChange);
				}
			}
		}
	}

	/**
	 * Helper method for initializing {@link DependentDiagramChange}s in the
	 * conflict report.
	 * 
	 * @param dependentDiagramChange
	 * @param modelChange
	 * @return
	 */
	protected DependentDiagramChange init(
			DependentDiagramChange dependentDiagramChange,
			DiffElement modelChange) {
		if (dependentDiagramChange == null) {
			dependentDiagramChange = ConflictReportFactory.eINSTANCE
					.createDependentDiagramChange();
			dependentDiagramChange.setModelChange(modelChange);
		}
		return dependentDiagramChange;
	}

	/**
	 * Finds {@link DependentDiagramChange}s in a given diagram.
	 * 
	 * @param modelChange
	 * @param diagramResourceSnapshot
	 * @return
	 */
	protected DependentDiagramChange findDependencies(DiffElement modelChange,
			ComparisonResourceSnapshot diagramResourceSnapshot) {
		DependentDiagramChange dependentDiagramChange = null;

		switch (modelChange.getKind()) {
		case ADDITION:
			for (DiffElement diff : diagramResourceSnapshot.getDiff()
					.getDifferences()) {
				if (!diff.getKind().equals(DifferenceKind.ADDITION)) {
					continue;
				}
				EObject leftElement = DiffUtil.getLeftElement(diff);
				if (leftElement instanceof View) {
					View viewElement = (View) leftElement;
					if (viewElement.getElement() != null
							&& EcoreUtil
									.getURI(viewElement.getElement())
									.fragment()
									.equals(EcoreUtil
											.getURI(DiffUtil
													.getLeftElement(modelChange))
											.fragment())) {
						dependentDiagramChange = init(dependentDiagramChange,
								modelChange);
						dependentDiagramChange.getDiagramChanges().add(diff);
					}
					if (viewElement instanceof Edge) {
						if (((Edge) viewElement).getSource() != null
								&& EcoreUtil
										.getURI(((Edge) viewElement)
												.getSource())
										.fragment()
										.equals(EcoreUtil
												.getURI(DiffUtil
														.getLeftElement(modelChange))
												.fragment())) {
							dependentDiagramChange = init(
									dependentDiagramChange, modelChange);
							dependentDiagramChange.getDiagramChanges()
									.add(diff);
						}
					}
				}
			}
			return dependentDiagramChange;
		case DELETION:
			for (DiffElement diff : diagramResourceSnapshot.getDiff()
					.getDifferences()) {
				if (!diff.getKind().equals(DifferenceKind.DELETION)) {
					continue;
				}
				EObject leftElement = DiffUtil.getLeftElement(diff);
				if (leftElement instanceof View) {
					View viewElement = (View) leftElement;
					if (viewElement.getElement() != null
							&& EcoreUtil
									.getURI(viewElement.getElement())
									.fragment()
									.equals(EcoreUtil
											.getURI(DiffUtil
													.getLeftElement(modelChange))
											.fragment())) {
						dependentDiagramChange = init(dependentDiagramChange,
								modelChange);
						dependentDiagramChange.getDiagramChanges().add(diff);
					}
				}
			}
			return dependentDiagramChange;
		case CHANGE:
			// TODO
			;
		case MOVE:
			// TODO
			;
		}

		return null;
	}
}
