/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.merge.conflictdiagram.util;

import static org.modelversioning.conflicts.profile.ModelversioningProfile.addAdd;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.conflict;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.deleteUpdate;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.modelversioningProfile;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.add;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.update;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.delete;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.myAdd;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.myCompositeChange;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.myDelete;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.myMove;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.myUpdate;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.operationContractViolation;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.overlappingChange;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.theirAdd;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.theirCompositeChange;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.theirDelete;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.theirMove;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.theirUpdate;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.updateUpdate;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.addForbid;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.deleteUse;
import static org.modelversioning.conflicts.profile.ModelversioningProfile.changeUse;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.compare.diff.metamodel.DiffElement;
import org.eclipse.emf.compare.diff.metamodel.DifferenceKind;
import org.eclipse.emf.compare.diff.metamodel.UpdateAttribute;
import org.eclipse.emf.compare.diff.metamodel.UpdateContainmentFeature;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.uml2.uml.Actor;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.DirectedRelationship;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLFactory;
import org.modelversioning.conflictreport.ConflictReport;
import org.modelversioning.conflictreport.EquivalentChange;
import org.modelversioning.conflictreport.conflict.AddAdd;
import org.modelversioning.conflictreport.conflict.ConditionViolation;
import org.modelversioning.conflictreport.conflict.Conflict;
import org.modelversioning.conflictreport.conflict.DeleteUpdate;
import org.modelversioning.conflictreport.conflict.DifferentBindingSize;
import org.modelversioning.conflictreport.conflict.MatchingNegativeApplicationCondition;
import org.modelversioning.conflictreport.conflict.MissingObject;
import org.modelversioning.conflictreport.conflict.OperationContractDiagnostics;
import org.modelversioning.conflictreport.conflict.OperationContractViolation;
import org.modelversioning.conflictreport.conflict.OverlappingChange;
import org.modelversioning.conflictreport.conflict.UpdateUpdate;
import org.modelversioning.conflictreport.conflict.ViolatedPrecondition;
import org.modelversioning.conflicts.profile.ModelversioningProfile;
import org.modelversioning.conflicts.profile.ModelversioningProfile.State;
import org.modelversioning.conflicts.profile.util.ProfileUtil;
import org.modelversioning.core.conditions.Condition;
import org.modelversioning.core.conditions.templatebindings.TemplateBinding;
import org.modelversioning.core.conditions.templatebindings.TemplateBindingCollection;
import org.modelversioning.core.diff.util.DiffUtil;
import org.modelversioning.merge.IMergeStrategy;
import org.modelversioning.operations.detection.operationoccurrence.OperationOccurrence;

/**
 * Utility Class for annotating differences in UML models.
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */

public class MergeUtil {

	/**
	 * A map holding the URI fragments of the merged {@link EObject EObjects}.
	 */
	private Map<String, EObject> uriFragmentMap = null;

	/**
	 * A map holding the diffElements and the corresponding change stereotypes.
	 */
	private Map<DiffElement, EObject> diffChangeStereotypeMap = null;

	/**
	 * A map holding the collaborations for the corresponding operation
	 * occurrences.
	 */
	private Map<DiffElement, NamedElement> operationCollaborationMap = null;

	/**
	 * Constructor
	 */
	public MergeUtil(Resource mergedModel) {
		// init URI Map
		uriFragmentMap = new HashMap<String, EObject>();
		for (EObject o : mergedModel.getContents()) {
			initFragmentMap(o);
		}

		diffChangeStereotypeMap = new HashMap<DiffElement, EObject>();
		operationCollaborationMap = new HashMap<DiffElement, NamedElement>();
	}

	/**
	 * Populates the <code>uriFragmentMap</code>.
	 * 
	 * @param rootObject
	 */
	private void initFragmentMap(EObject rootObject) {
		TreeIterator<EObject> i = rootObject.eAllContents();
		uriFragmentMap.put(EcoreUtil.getURI(rootObject).fragment(), rootObject);
		while (i.hasNext()) {
			EObject o = i.next();
			uriFragmentMap.put(EcoreUtil.getURI(o).fragment(), o);
		}
	}

	/**
	 * Utility method to ensure the profile application on given package
	 * element.
	 * 
	 * @param package The element to check.
	 */
	public void ensureProfileApplication(Package package_) {
		if (package_.getAppliedProfile(
				ModelversioningProfile.MODELVERSIONING_PROFILE_NAME, true) == null)
			package_.applyProfile(ModelversioningProfile.modelversioningProfile);
	}

	/**
	 * Utility method to apply change stereotypes.
	 * 
	 * @param diff
	 * @param mergeStrategy
	 * @param conflictReport
	 * @param side
	 * @param user
	 */
	public void applyChangeStereotype(DiffElement diff,
			IMergeStrategy mergeStrategy, ConflictReport conflictReport,
			Side side, Actor user, DiagramMergeUtil diagramUtil) {

		State state = mergeStrategy.shouldMerge(diff, conflictReport) ? State.APPLIED
				: State.PENDING;
		NamedElement changedElement = null;
		EObject changedEObject = null;
		Stereotype stereotype = getStereotypeForDiff(diff, side);

		if (diff instanceof OperationOccurrence) {
			Package rootContainer = null;
			EList<Dependency> dependencies = new BasicEList<Dependency>();

			for (DiffElement hiddenChange : ((OperationOccurrence) diff)
					.getHiddenChanges()) {
				if (rootContainer == null) {
					rootContainer = (Package) getCounterpart(EcoreUtil.getURI(
							((Element) getElement(hiddenChange))
									.getNearestPackage()).fragment());
				} else {
					rootContainer = (Package) getCounterpart(EcoreUtil.getURI(
							getNearestCommonContainer(getElement(hiddenChange),
									rootContainer)).fragment());
				}
			}

			changedElement = rootContainer.createPackagedElement(
					((OperationOccurrence) diff).getAppliedOperationName(),
					UMLFactory.eINSTANCE.createCollaboration().eClass());
			operationCollaborationMap.put(diff, changedElement);
			changedEObject = changedElement;

			for (TemplateBinding preCondition : ((OperationOccurrence) diff)
					.getPreConditionBinding().getBindings()) {
				for (EObject binding : preCondition.getEObjects()) {
					NamedElement target = (NamedElement) getCounterpart(EcoreUtil
							.getURI(getType(binding, NamedElement.class))
							.fragment());

					if (target == null) {
						return;
					}

					Dependency dependency = null;

					if (target instanceof Property && target.getOwner() != null
							&& target.getOwner() instanceof NamedElement) {
						dependency = ((Collaboration) changedElement)
								.createDependency((NamedElement) target
										.getOwner());
					} else {
						dependency = ((Collaboration) changedElement)
								.createDependency(target);
					}

					if (dependency != null) {
						dependency.setName(preCondition.getTemplateName());
						dependencies.add(dependency);
					}
				}
			}

			// create Shapes
			diagramUtil.createShapes((Collaboration) changedElement);
		} else {
			// get counterpart object
			changedEObject = getCounterpart(diff);
		}

		if (changedEObject == null
				&& conflictReport.getEquivalentChanges(diff) != null
				&& conflictReport.getEquivalentChanges(diff).size() > 0) {
			for (DiffElement equivalentChange : conflictReport
					.getEquivalentChanges(diff)) {
				if (conflictReport.getEquivalentPreferredChanges().contains(
						equivalentChange)) {
					changedEObject = getCounterpart(equivalentChange);
					break;
				}
			}
		}

		if (!(changedEObject instanceof NamedElement)) {
			return;
		}
		changedElement = (NamedElement) changedEObject;
		applyChangeStereotype(diff, user, state, changedElement, stereotype);

		if (diff instanceof UpdateAttribute) {
			UpdateAttribute updateAtt = (UpdateAttribute) diff;
			changedElement.setValue(stereotype,
					ProfileUtil.UPDATE_UPDATED_FEATURE_FEATURE,
					updateAtt.getAttribute());
			changedElement.setValue(stereotype,
					ProfileUtil.UPDATE_UPDATED_VALUE_FEATURE, updateAtt
							.getLeftElement().eGet(updateAtt.getAttribute()));
			changedElement.setValue(stereotype,
					ProfileUtil.UPDATE_OLD_VALUE_FEATURE, updateAtt
							.getRightElement().eGet(updateAtt.getAttribute()));
		}
		if (diff instanceof UpdateContainmentFeature) {
			UpdateContainmentFeature updateContainment = (UpdateContainmentFeature) diff;
			// changedElement.setValue(myUpdate, "updatedFeature",
			// updateContainment);
			// changedElement.setValue(myMove, "updatedValue",
			// updateContainment.getLeftTarget().toString());
			// changedElement.setValue(myMove, "oldValue",
			// updateContainment.getRightTarget().toString());
		}
	}

	/**
	 * Utility method to apply change stereotypes.
	 * 
	 * @param diff
	 * @param user
	 * @param state
	 * @param changedElement
	 * @param stereotype
	 */
	private void applyChangeStereotype(DiffElement diff, Actor user,
			State state, Element changedElement, Stereotype stereotype) {
		if (changedElement.getApplicableStereotypes().contains(stereotype)
				&& !changedElement.getAppliedStereotypes().contains(stereotype)) {

			changedElement.applyStereotype(stereotype);
			changedElement.setValue(stereotype,
					ProfileUtil.CHANGE_DIFF_ELEMENT_FEATURE, diff);
			changedElement.setValue(stereotype,
					ProfileUtil.CHANGE_APPLICATION_STATE_FEATURE,
					org.modelversioning.conflicts.profile.util.ProfileUtil
							.getApplicationState(state));
			changedElement.setValue(stereotype,
					ProfileUtil.CHANGE_USER_FEATURE, user);

			diffChangeStereotypeMap.put(diff,
					changedElement.getStereotypeApplication(stereotype));
		}
	}

	/**
	 * Utility method to apply equivalent change stereotypes.
	 * 
	 * @param equivalentChange
	 */
	public void applyEquivalentChangeStereotype(
			EquivalentChange equivalentChange) {
		DiffElement leftChange = equivalentChange.getLeftChange();
		DiffElement rightChange = equivalentChange.getRightChange();

		Element equivalentChangeElement = (Element) getCounterpart(equivalentChange
				.getPreferredChange());
		Stereotype stereotype = ModelversioningProfile.equivalentChange;

		if (equivalentChangeElement.isStereotypeApplicable(stereotype)
				&& !equivalentChangeElement.isStereotypeApplied(stereotype)) {
			equivalentChangeElement.applyStereotype(stereotype);

			equivalentChangeElement.setValue(stereotype,
					ProfileUtil.CONFLICT_THEIR_CHANGE_FEATURE,
					diffChangeStereotypeMap.get(leftChange));
			equivalentChangeElement.setValue(stereotype,
					ProfileUtil.CONFLICT_MY_CHANGE_FEATURE,
					diffChangeStereotypeMap.get(rightChange));
		}
	}

	/**
	 * Utility method to apply conflict stereotypes.
	 * 
	 * @param conflict
	 * @param diagramUtil
	 */
	public void applyConflictStereotype(Conflict conflict,
			DiagramMergeUtil diagramUtil) {

		DiffElement leftChange = conflict.getLeftChange();
		DiffElement rightChange = conflict.getRightChange();

		// FIXME this is only workaround for avoiding nullpointer exceptions
		if (leftChange instanceof OperationOccurrence) {
			leftChange = ((OperationOccurrence) conflict.getLeftChange())
					.getHiddenChanges().get(0);
		}
		if (rightChange instanceof OperationOccurrence) {
			rightChange = ((OperationOccurrence) conflict.getRightChange())
					.getHiddenChanges().get(0);
		}

		NamedElement conflictElement = null;
		Stereotype stereotype = null;

		// annotate Element
		if (conflict instanceof OverlappingChange
				&& !(conflict instanceof AddAdd)) {

			conflictElement = (NamedElement) getCounterpart(EcoreUtil.getURI(
					getNearestCommonContainer(getElement(leftChange),
							getElement(rightChange))).fragment());

			if (conflictElement != null) {
				if (conflict instanceof UpdateUpdate) {
					stereotype = updateUpdate;

				} else if (conflict instanceof DeleteUpdate) {
					stereotype = deleteUpdate;

				}
			}
		}
		// insert Collaboration for AddAdd conflict
		else if (conflict instanceof AddAdd) {
			EObject leftChangedElement = getElement(leftChange);
			EObject rightChangedElement = getElement(rightChange);

			Package rootContainer = null;

			if (leftChangedElement != null && rightChangedElement != null) {
				rootContainer = (Package) getCounterpart(EcoreUtil.getURI(
						getNearestCommonPackage(leftChangedElement,
								rightChangedElement)).fragment());
			} else if (leftChangedElement != null) {
				rootContainer = (Package) ((Element) getCounterpart(EcoreUtil
						.getURI(leftChangedElement).fragment()))
						.getNearestPackage();
			} else if (rightChangedElement != null) {
				rootContainer = (Package) ((Element) getCounterpart(EcoreUtil
						.getURI(rightChangedElement).fragment()))
						.getNearestPackage();
			}

			conflictElement = rootContainer.createPackagedElement(conflict
					.getClass().getSimpleName(), UMLFactory.eINSTANCE
					.createCollaboration().eClass());
			NamedElement leftTarget = (NamedElement) getCounterpart(leftChange);
			NamedElement rightTarget = (NamedElement) getCounterpart(rightChange);

			Dependency leftDependency = ((Collaboration) conflictElement)
					.createDependency(leftTarget);
			EList<Dependency> leftDependencies = new BasicEList<Dependency>();
			leftDependencies.add(leftDependency);

			Dependency rightDependency = ((Collaboration) conflictElement)
					.createDependency(rightTarget);
			EList<Dependency> rightDependencies = new BasicEList<Dependency>();
			rightDependencies.add(rightDependency);

			// create Shapes
			diagramUtil.createShapes(leftTarget, rightTarget,
					(Collaboration) conflictElement, leftDependencies,
					rightDependencies);

			stereotype = addAdd;
		} else if (conflict instanceof OperationContractViolation) {
			DiffElement operationOccurrence = null;
			if (conflict.getLeftChange() instanceof OperationOccurrence) {
				operationOccurrence = conflict.getLeftChange();
			} else if (conflict.getRightChange() instanceof OperationOccurrence) {
				operationOccurrence = conflict.getRightChange();
			}
			conflictElement = operationCollaborationMap
					.get(operationOccurrence);
			stereotype = operationContractViolation;

			OperationContractViolation opViolation = (OperationContractViolation) conflict;
			OperationContractDiagnostics diagnostics = opViolation
					.getDiagnostics();

			// if (diagnostics instanceof MatchingNegativeApplicationCondition)
			// {
			// MatchingNegativeApplicationCondition matchingNacDiagnostics =
			// (MatchingNegativeApplicationCondition) diagnostics;
			// TemplateBindingCollection templateBindingCollection =
			// matchingNacDiagnostics
			// .getNacBinding();
			// EList<TemplateBinding> templateBindings =
			// templateBindingCollection
			// .getBindings();
			// for (TemplateBinding binding : templateBindings) {
			// for (EObject boundObject : binding.getEObjects()) {
			// if (boundObject instanceof Element) {
			// // annotate nac binding
			// // TODO
			// }
			// }
			// }
			// } else
			if (diagnostics instanceof ViolatedPrecondition) {
				ViolatedPrecondition violatedPreconditionDiagnostics = (ViolatedPrecondition) diagnostics;
				EList<ConditionViolation> conditionViolations = violatedPreconditionDiagnostics
						.getConditionViolations();
				for (ConditionViolation conditionViolation : conditionViolations) {
					EObject violatingObject = conditionViolation
							.getViolatingObject();
					if (violatingObject instanceof Element) {
						Condition condition = conditionViolation
								.getViolatedCondition();
						// annotate precondition violation
						for (Dependency dependency : ((Collaboration) conflictElement)
								.getClientDependencies()) {
							if (/*(dependency.getSuppliers().contains(
									getCounterpart(EcoreUtil.getURI(
											violatingObject).fragment()))
									|| dependency.getSuppliers().contains(
											getCounterpart(EcoreUtil.getURI(
													((Element) violatingObject).getOwner()).fragment())))
									&&*/ 
									dependency.getName().equals(condition.getTemplate().getName())
									&& dependency
											.isStereotypeApplicable(addForbid)
									&& !dependency
											.isStereotypeApplied(addForbid)) {
								dependency.applyStereotype(addForbid);
								Stereotype deleteUseApplication = dependency
										.getAppliedStereotype(addForbid
												.getQualifiedName());
								dependency.setValue(deleteUseApplication,
										"violatedCondition", condition);
							}
						}
					}
				}
			} else if (diagnostics instanceof MissingObject) {
				MissingObject missingObjectDiagnostics = (MissingObject) diagnostics;
				EObject missingObject = missingObjectDiagnostics
						.getMissingObject();
				if (missingObject instanceof Element) {
					// annotate missing object
					for (Dependency dependency : ((Collaboration) conflictElement)
							.getClientDependencies()) {
						if (/*(dependency.getSuppliers().contains(
								getCounterpart(EcoreUtil.getURI(
								violatingObject).fragment()))
						|| dependency.getSuppliers().contains(
								getCounterpart(EcoreUtil.getURI(
										((Element) violatingObject).getOwner()).fragment())))
						&&*/ 
						dependency.getName().equals(missingObjectDiagnostics.getTemplate().getName())
								&& dependency.isStereotypeApplicable(deleteUse)
								&& !dependency.isStereotypeApplied(deleteUse)) {
							dependency.applyStereotype(deleteUse);
							Stereotype deleteUseApplication = dependency
									.getAppliedStereotype(deleteUse
											.getQualifiedName());
							dependency.setValue(deleteUseApplication,
									"violatedCondition",
									missingObjectDiagnostics);
						}
					}
				}
			}
		}

		if (conflictElement.getApplicableStereotypes().contains(stereotype)) {
			if (!conflictElement.isStereotypeApplied(stereotype)) {
				conflictElement.applyStereotype(stereotype);
			}

			// set additional TaggedValues for stereotypes
			conflictElement.setValue(stereotype,
					ProfileUtil.CONFLICT_IS_RESOLVED_FEATURE, false);

			if (diffChangeStereotypeMap.containsKey(leftChange)) {
				EList<EObject> theirChanges = new BasicEList<EObject>();
				theirChanges.addAll((EList<EObject>) ProfileUtil.getValue(
						conflictElement, stereotype,
						ProfileUtil.CONFLICT_THEIR_CHANGE_FEATURE));
				theirChanges.add(diffChangeStereotypeMap.get(leftChange));

				conflictElement
						.setValue(stereotype,
								ProfileUtil.CONFLICT_THEIR_CHANGE_FEATURE,
								theirChanges);
			}
			if (diffChangeStereotypeMap.containsKey(rightChange)) {
				EList<EObject> myChanges = new BasicEList<EObject>();
				myChanges.addAll((EList<EObject>) ProfileUtil.getValue(
						conflictElement, stereotype,
						ProfileUtil.CONFLICT_MY_CHANGE_FEATURE));
				myChanges.add(diffChangeStereotypeMap.get(rightChange));

				conflictElement.setValue(stereotype,
						ProfileUtil.CONFLICT_MY_CHANGE_FEATURE, myChanges);
			}
		}

		if (conflictElement instanceof Collaboration) {
			diagramUtil.updateShapes((Collaboration)conflictElement);
		}
	}

	public void updateChangeStereotype(DiffElement diff,
			IMergeStrategy mergeStrategy, ConflictReport conflictReport,
			Side side, Actor user) {
		State state = mergeStrategy.shouldMerge(diff, conflictReport) ? State.APPLIED
				: State.REVERTED;
		Element changedElement = null;
		Stereotype stereotype = getStereotypeForDiff(diff, side);

		// get counterpart object
		EObject changedEObject = getCounterpart(diff);
		if (!(changedEObject instanceof Element)) {
			return;
		}

		changedElement = (Element) changedEObject;

		if (changedElement.isStereotypeApplied(stereotype)) {
			changedElement.setValue(stereotype,
					ProfileUtil.CHANGE_APPLICATION_STATE_FEATURE,
					ProfileUtil.getApplicationState(state));
		}
	}

	public void updateConflictStereotype(DiffElement diff,
			Stereotype stereotype, boolean isResolved) {
		Element conflictElement = null;
		EObject conflictEObject = null;
		Stereotype conflictStereotype = (Stereotype) EcoreUtil.resolve(
				stereotype, modelversioningProfile);

		// get counterpart object
		if (ProfileUtil.conformsTo(stereotype, overlappingChange)
				&& !ProfileUtil.conformsTo(stereotype, addAdd)) {
			conflictEObject = getCounterpart(diff);
		} else {
			EObject changeEObject = getCounterpart(diff);

			if (!(changeEObject instanceof Element)) {
				return;
			}
			Element changeElement = (Element) changeEObject;

			for (DirectedRelationship relationship : changeElement
					.getTargetDirectedRelationships()) {
				if (relationship instanceof Dependency) {
					Dependency dependency = (Dependency) relationship;
					for (Element srcElement : dependency.getSources()) {
						if (srcElement instanceof Collaboration) {
							Collaboration collaboration = (Collaboration) srcElement;
							if (collaboration
									.getAppliedStereotype(conflictStereotype
											.getQualifiedName()) != null) {
								conflictEObject = collaboration;
								break;
							}
						}
					}
				}
			}
		}

		if (!(conflictEObject instanceof Element)) {
			return;
		}
		conflictElement = (Element) conflictEObject;
		if (!conflictElement.isStereotypeApplied(conflictStereotype)) {
			EList<Stereotype> appliedConflict = conflictElement
					.getAppliedSubstereotypes(conflict);
			if (appliedConflict != null && appliedConflict.size() > 0) {
				conflictStereotype = appliedConflict.get(0);
			} else {
				return;
			}
		}
		conflictElement.setValue(conflictStereotype, "isResolved", isResolved);
	}

	private Stereotype getStereotypeForDiff(DiffElement diff, Side side) {
		Stereotype stereotype = null;

		if (diff instanceof OperationOccurrence) {
			stereotype = side.equals(Side.RIGHT) ? myCompositeChange
					: theirCompositeChange;
		}
		if (diff.getKind().equals(DifferenceKind.ADDITION)) {
			stereotype = side.equals(Side.RIGHT) ? myAdd : theirAdd;
		} else if (diff.getKind().equals(DifferenceKind.CHANGE)) {
			stereotype = side.equals(Side.RIGHT) ? myUpdate : theirUpdate;
		} else if (diff.getKind().equals(DifferenceKind.MOVE)) {
			stereotype = side.equals(Side.RIGHT) ? myMove : theirMove;
		} else if (diff.getKind().equals(DifferenceKind.DELETION)) {
			stereotype = side.equals(Side.RIGHT) ? myDelete : theirDelete;
		}

		return stereotype;
	}

	/**
	 * Finds the nearest container <code>element1</code> and
	 * <code>element2</code> have in common.
	 * 
	 * @param element1
	 *            The first element.
	 * @param element2
	 *            The second element.
	 * @return nearest common container element, or null if there is no common
	 *         container.
	 */
	public static EObject getNearestCommonContainer(EObject element1,
			EObject element2) {

		if (EcoreUtil.getURI(element1).fragment()
				.equals(EcoreUtil.getURI(element2).fragment()))
			return element1;

		EList<EObject> container1 = new BasicEList<EObject>();
		EList<EObject> container2 = new BasicEList<EObject>();

		container1.add(element1);
		container2.add(element2);

		getContainerList(element1, container1);
		getContainerList(element2, container2);

		if (container1.size() < container2.size())
			return getNearestCommonContainer(container1, container2,
					EObject.class);
		else
			return getNearestCommonContainer(container2, container1,
					EObject.class);
	}

	/**
	 * Finds the nearest container <code>element1</code> and
	 * <code>element2</code> have in common.
	 * 
	 * @param element1
	 *            The first element.
	 * @param element2
	 *            The second element.
	 * @return nearest common container element, or null if there is no common
	 *         container.
	 */
	public static <T extends EObject> T getNearestCommonContainer(
			EObject element1, EObject element2, Class<T> containerType) {

		if (EcoreUtil.getURI(element1).fragment()
				.equals(EcoreUtil.getURI(element2).fragment())
				&& containerType.isAssignableFrom(element1.getClass())) {
			return (T) element1;
		}

		EList<EObject> container1 = new BasicEList<EObject>();
		EList<EObject> container2 = new BasicEList<EObject>();

		container1.add(element1);
		container2.add(element2);

		getContainerList(element1, container1);
		getContainerList(element2, container2);

		if (container1.size() < container2.size())
			return getNearestCommonContainer(container1, container2,
					containerType);
		else
			return getNearestCommonContainer(container2, container1,
					containerType);
	}

	/**
	 * Finds the nearest common package of <code>element1</code> and
	 * <code>element2</code>.
	 * 
	 * @param element1
	 *            The first element.
	 * @param element2
	 *            The second element.
	 * @return nearest common package, or null if there is no common package.
	 */
	public static Package getNearestCommonPackage(EObject element1,
			EObject element2) {

		if (element1 instanceof Package
				&& EcoreUtil.getURI(element1).fragment()
						.equals(EcoreUtil.getURI(element2).fragment()))
			return (Package) element1;

		EList<EObject> container1 = new BasicEList<EObject>();
		EList<EObject> container2 = new BasicEList<EObject>();

		container1.add(element1);
		container2.add(element2);

		getContainerList(element1, container1);
		getContainerList(element2, container2);

		if (container1.size() < container2.size())
			return getNearestCommonContainer(container1, container2,
					Package.class);
		else
			return getNearestCommonContainer(container2, container1,
					Package.class);
	}

	/**
	 * Finds the first common element, the lists <code>container1</code> and
	 * <code>container2</code> have in common.
	 * 
	 * @param container1
	 *            The first list of elements.
	 * @param container2
	 *            The second list of elements.
	 * @return
	 * @return first common container element, or null if there is no common
	 *         container.
	 */
	public static <T extends EObject> T getNearestCommonContainer(
			EList<EObject> container1, EList<EObject> container2,
			Class<T> containerType) {

		Iterator<EObject> i1 = container1.iterator();
		while (i1.hasNext()) {
			EObject o1 = i1.next();

			if (containerType.isAssignableFrom(o1.getClass())) {
				Iterator<EObject> i2 = container2.iterator();

				while (i2.hasNext()) {
					EObject o2 = i2.next();
					if (EcoreUtil.getURI(o1).fragment()
							.equals(EcoreUtil.getURI(o2).fragment())) {
						return (T) o1;
					}
				}
			}
		}
		return null;
	}

	/**
	 * Returns the list of container elements for a specific element.
	 * 
	 * @param element
	 *            The element to find containers for.
	 * @param containerList
	 *            The list of container elements.
	 * @return list of container elements.
	 */
	public static EList<EObject> getContainerList(EObject element,
			EList<EObject> containerList) {
		EObject containerObject = element.eContainer();
		if (containerObject != null) {
			containerList.add(containerObject);
			getContainerList(containerObject, containerList);
		}

		return containerList;
	}

	/**
	 * Returns the counterpart for the element with the specified uri fragment.
	 * 
	 * @param uriFragment
	 *            The uri fragment of the element in question.
	 * @return the counterpart element.
	 */
	public EObject getCounterpart(String uriFragment) {
		return uriFragmentMap.get(uriFragment);
	}

	/**
	 * Returns the counterpart element of the affected element of a diff element
	 * 
	 * @param diff
	 *            The diff element.
	 * @return the counterpart of the changed element.
	 */
	public EObject getCounterpart(DiffElement diff) {
		return getCounterpart(EcoreUtil.getURI(getElement(diff)).fragment());
	}

	/**
	 * Returns the affected element of a diff element
	 * 
	 * @param diff
	 *            The diff element.
	 * @return the changed element.
	 */
	public EObject getElement(DiffElement diff) {
		if (diff.getKind().equals(DifferenceKind.ADDITION)) {
			return DiffUtil.getLeftElement(diff);
		} else {
			return DiffUtil.getRightElement(diff);
		}
	}

	/**
	 * Returns the stereotype applications of all changes of an element
	 * 
	 * @param element
	 *            The element to find changes for
	 * @return change stereotype applications
	 */
	public EList<EObject> getChanges(Element element) {
		EList<EObject> appliedChanges = new BasicEList<EObject>();

		for (Stereotype stereotype : element.getAppliedStereotypes()) {
			Stereotype resolvedStereotype = stereotype;

			if (stereotype.eIsProxy()) {
				resolvedStereotype = (Stereotype) EcoreUtil.resolve(stereotype,
						ModelversioningProfile.modelversioningProfile);
			}

			if (ProfileUtil.conformsTo(resolvedStereotype,
					ModelversioningProfile.change)) {
				// element.isStereotypeApplied(stereotype)
				appliedChanges
						.add(element.getStereotypeApplication(stereotype));
			}
		}

		return appliedChanges;
	}

	/**
	 * Checks a given object for type conformance. Goes up the containment
	 * hierarchy and returns the nearest container, which is an instance of the
	 * given type.
	 * 
	 * @param object
	 *            The context object.
	 * @param containerType
	 *            The type to look for.
	 * @return first container element of the given type, or null if there is
	 *         none.
	 */
	public static <T extends EObject> T getType(EObject object,
			Class<T> containerType) {
		while (object != null) {
			if (containerType.isInstance(object)) {
				return (T) object;
			}

			object = object.eContainer();
		}

		return null;
	}
}
