/**
 * <copyright>
 *
 * Copyright (c) 2010 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.conflicts.profile.util;

/**
 * Utility Class for working with the {@link ModelversioningProfile}
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */

import static org.modelversioning.conflicts.profile.ModelversioningProfile.state;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.impl.EEnumLiteralImpl;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Stereotype;
import org.modelversioning.conflicts.profile.ModelversioningProfile.State;

public class ProfileUtil {

	public static final String CHANGE_APPLICATION_STATE_FEATURE = "state";
	public static final String CHANGE_DIFF_ELEMENT_FEATURE = "changeElement";
	public static final String CHANGE_USER_FEATURE = "user";
	
	public static final String UPDATE_UPDATED_FEATURE_FEATURE = "updatedFeature";
	public static final String UPDATE_UPDATED_VALUE_FEATURE = "updatedValue";
	public static final String UPDATE_OLD_VALUE_FEATURE = "oldValue";
	
	public static final String CONFLICT_MY_CHANGE_FEATURE = "myChange";
	public static final String CONFLICT_THEIR_CHANGE_FEATURE = "theirChange";
	public static final String CONFLICT_IS_RESOLVED_FEATURE = "isResolved";
	
	/**
	 * Returns a tagged value of the {@link Stereotype} applied to the given {@link Element}
	 * @param element
	 * 			The element the stereotype is applied to.
	 * @param stereotype
	 * 			The stereotype. 
	 * @param taggedValue
	 * 			The tagged value to look for.
	 * @return The value of a stereotype's tagged value.
	 */
	public static Object getValue(Element element, Stereotype stereotype, String taggedValue) {
		return element.getValue(stereotype, taggedValue);
	}
	
	/**
	 * Reflexively determines a feature's value of a stereotype application hold as {@link DynamicEObjectImpl}.
	 * @param stereotypeApplication
	 * 			The stereotype application.
	 * @param feature
	 * 			The feature to get the value for.
	 * @return The value of a stereotype application's feature.
	 */
	public static Object getFeature(EObject stereotypeApplication, String feature) {
		EStructuralFeature structuralFeature = stereotypeApplication.eClass().getEStructuralFeature(feature);
		if (structuralFeature != null) {
			return stereotypeApplication.eGet(structuralFeature);
		}
		
		return null;
	}
	
	/**
	 * Determines the application state of a change stereotype application.
	 * 
	 * @param changedObject
	 * @return true, if the application state is APPLIED, false otherwise.
	 */
	public static boolean isApplied(EObject changeObject) {
		EStructuralFeature applicationStateFeature = changeObject.eClass()
				.getEStructuralFeature(CHANGE_APPLICATION_STATE_FEATURE);

		if (applicationStateFeature != null) {
			EEnumLiteralImpl literal = (EEnumLiteralImpl) changeObject
					.eGet(applicationStateFeature);
			if (literal.getName().equals(State.APPLIED.name())) {
				return true;
			}
		}

		return false;
	}
	
	/**
	 * Determines the application state of a change stereotype application.
	 * 
	 * @param changedObject
	 * @return true, if the application state is REVERTED, false otherwise.
	 */
	public static boolean isReverted(EObject changeObject) {
		EStructuralFeature applicationStateFeature = changeObject.eClass()
				.getEStructuralFeature(CHANGE_APPLICATION_STATE_FEATURE);

		if (applicationStateFeature != null) {
			EEnumLiteralImpl literal = (EEnumLiteralImpl) changeObject
					.eGet(applicationStateFeature);
			if (literal.getName().equals(State.REVERTED.name())) {
				return true;
			}
		}

		return false;
	}
	
	/**
	 * Determines the application state of a change stereotype application.
	 * 
	 * @param changedObject
	 * @return true, if the application state is REVERTED, false otherwise.
	 */
	public static boolean isPending(EObject changeObject) {
		EStructuralFeature applicationStateFeature = changeObject.eClass()
				.getEStructuralFeature(CHANGE_APPLICATION_STATE_FEATURE);

		if (applicationStateFeature != null) {
			EEnumLiteralImpl literal = (EEnumLiteralImpl) changeObject
					.eGet(applicationStateFeature);
			if (literal.getName().equals(State.PENDING.name())) {
				return true;
			}
		}

		return false;
	}
	
	/**
	 * Searches the inheritance hierarchy of the given stereotype for a
	 * stereotype with the given general name.
	 * 
	 * @param stereotype
	 *            The stereotype to inspect.
	 * @param general
	 *            The name of the general stereotype to look for.
	 * @return The general stereotype if available. Null otherwise.
	 */
	public static Stereotype getGeneral(Stereotype stereotype, String general) {
		Classifier generalClassifier = stereotype.getGeneral(general);
		if (generalClassifier != null
				&& generalClassifier instanceof Stereotype) {
			return (Stereotype) generalClassifier;
		} else {
			for (Classifier classifier : stereotype.getGenerals()) {
				if (classifier instanceof Stereotype)
					return getGeneral((Stereotype) classifier, general);
			}
		}
		return null;
	}
	
	public static boolean isConflict(Stereotype stereotype) {
		if (getGeneral(stereotype, "Conflict") != null) {
			return true;
		}
		return false;
	}
	
	public static boolean isChange(Stereotype stereotype) {
		if (getGeneral(stereotype, "Change") != null) {
			return true;
		}
		return false;
	}
	
	public static boolean conformsTo(Stereotype stereotype, Stereotype other) {
		if (stereotype.getQualifiedName().equals(other.getQualifiedName())) {
			return true;
		}
		else {
			for (Classifier classifier : stereotype.allParents()) {
				if (classifier.getQualifiedName().equals(other.getQualifiedName())) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Converts a {@link State} to a {@link EnumerationLiteral}
	 * @param s
	 * 		The state to convert.
	 * @return
	 * 		The converted state.
	 */
	public static EnumerationLiteral getApplicationState(State s) {
		return state.getOwnedLiteral(s.name());
	}
}
