/**
 * <copyright>
 *
 * Copyright (c) 2011 modelversioning.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */

package org.modelversioning.conflicts.detection.impl;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.modelversioning.conflictreport.EquivalentChange;
import org.modelversioning.conflictreport.conflict.AddAdd;
import org.modelversioning.conflictreport.conflict.Conflict;
import org.modelversioning.conflictreport.conflict.UpdateUpdate;
import org.modelversioning.conflicts.detection.IThreeWayDiffProvider;
import org.modelversioning.conflicts.detection.engine.IOverlappingChangeDetector;

/**
 * {@link IOverlappingChangeDetector} for detecting {@link UpdateUpdate}
 * Conflicts in UML models.
 * 
 * <p>
 * This detector detects an {@link AddAdd} conflict. {@link AddAdd} conflicts
 * occur if added model elements have the same name within a namespace but have
 * different features set. Deep equal model elements are ignored and added to
 * the list of <code>equivalentChanges</code>.
 * </p>
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class UMLAddAddConflictDetector implements IOverlappingChangeDetector {

	private static final String NAME = "Add Add Conflict Detector";

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.modelversioning.conflicts.detection.engine.IConflictDetector#getId()
	 */
	@Override
	public String getId() {
		return "org.modelversioning.conflicts.detection.uml.addAdd";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.modelversioning.conflicts.detection.engine.IConflictDetector#
	 * getTargetModelNsURI()
	 */
	@Override
	public String getTargetModelNsURI() {
		return "http://www.eclipse.org/uml2/3.0.0/UML";
	}

	@Override
	public String getName() {
		return NAME;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * Detects concurrent add of the same {@link EObject}.
	 */
	@Override
	public void detectOverlappingChanges(IThreeWayDiffProvider threeWayDiff,
			EList<Conflict> conflicts,
			EList<EquivalentChange> equivalentChanges, IProgressMonitor monitor) {
		EList<EObject> leftAddedEObjects = threeWayDiff.getAddedEObjects(
				Side.LEFT, false);
		EList<EObject> rightAddedEObjects = threeWayDiff.getAddedEObjects(
				Side.RIGHT, false);
		monitor.beginTask("Searching for Add Add conflicts",
				leftAddedEObjects.size());
		try {
			for (EObject leftAddObject : leftAddedEObjects) {
				for (EObject rightAddObject : rightAddedEObjects) {
					if (leftAddObject.eClass().equals(rightAddObject.eClass())
							&& (threeWayDiff
									.getMatchingEObject(
											leftAddObject.eContainer(),
											Side.LEFT, true)
									.equals(threeWayDiff.getMatchingEObject(
											rightAddObject.eContainer(),
											Side.RIGHT, true)))) {
						EStructuralFeature nameFeature = leftAddObject.eClass()
								.getEStructuralFeature("name");
						if (nameFeature != null) {
							Object leftName = leftAddObject.eGet(nameFeature);
							Object rightName = rightAddObject.eGet(nameFeature);

							if (leftName != null && rightName != null && leftName.toString()
									.equals(rightName.toString())) {
								boolean equal = true;

								// check features of the eObjects and
								// decide if they are equal
								EList<EAttribute> attributes = leftAddObject
										.eClass().getEAllAttributes();
								for (EAttribute attribute : attributes) {
									if (!leftAddObject.eGet(attribute).equals(
											rightAddObject.eGet(attribute))) {
										equal = false;
										break;
									}
								}
								// TODO check also references

								if (equal) {
									EquivalentChange equivalentChange = CONFLICT_REPORT_FACTORY
											.createEquivalentChange();
									// it doesn't matter which side is preferred
									equivalentChange.setPreferSide(Side.RIGHT);
									equivalentChange.setLeftChange(threeWayDiff
											.getAddElement(leftAddObject,
													Side.LEFT));
									equivalentChange
											.setRightChange(threeWayDiff
													.getAddElement(
															rightAddObject,
															Side.RIGHT));
									equivalentChanges.add(equivalentChange);
								} else {
									AddAdd addAdd = CONFLICT_FACTORY
											.createAddAdd();
									addAdd.setLeftChange(threeWayDiff
											.getAddElement(leftAddObject,
													Side.LEFT));
									addAdd.setRightChange(threeWayDiff
											.getAddElement(rightAddObject,
													Side.RIGHT));
									conflicts.add(addAdd);
								}
							}
						}
					}
				}
				monitor.worked(1);
			}
		} finally {
			monitor.done();
		}
	}

	@Override
	public void initialize() {
		// noop
	}
}
