/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */
package org.modelevolution.multiview.diff.encoding.test;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.modelevolution.multiview.Lifeline;
import org.modelevolution.multiview.LifelineElement;
import org.modelevolution.multiview.Message;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.MultiviewPackage;
import org.modelevolution.multiview.ReceiveEvent;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.Transition;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;
import org.modelversioning.core.match.MatchException;

/**
 * @author <a href="mailto:widl@big.tuwien.ac.at">Magdalena Widl</a>
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class ModelVerifier {
	private static final Logger evalLogger = Logger
			.getLogger("org.modelevolution.multiview.diff.encoding.eval");

	public static final String FILE_EXTENSION = "mvml";
	public static final String DEFAULT_NAME = "DEFAULT";
	private static final boolean DEBUG = false;

	public ModelVerifier() {
	}

	/* FIXME this works only for deterministic state machines! */
	public boolean verify(File file) {

		// logger.log(Level.INFO,
		// "Starting ModelVerifier for {0}.", file.getPath());
		boolean valid = false;
		ResourceSet resourceSet = null;
		Resource resource = null;
		MultiviewModel mmmodel = null;
		HashMap<Lifeline, EList<Message>> lifelineToMessages = new HashMap<Lifeline, EList<Message>>();

		resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(FILE_EXTENSION, new UUIDResourceFactoryImpl());

		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put("xmi", new UUIDResourceFactoryImpl());

		URI fileURI_id = URI.createFileURI(file.getAbsolutePath());
		resource = resourceSet.getResource(fileURI_id, true);
		mmmodel = (MultiviewModel) resource.getContents().get(0);

//		for (Message m : mmmodel.getSequenceview().getOrderedMessages()) {
//			System.out.println(m.getBody().getName()+"\tSND:\t"+m.getSender().getName()+"\tRCV:\t"+m.getReceiver().getName());
//		}

		lifelineToMessages = getMap(mmmodel);

		valid = true;

		for (Lifeline l : lifelineToMessages.keySet()) {

			if (l.getClass_() == null
					|| l.getClass_().getStatemachine() == null) {
				l.initDummyStatemachine();
			}

			Region sm = l.getClass_().getStatemachine();

			EList<State> startStates = getStartStates(sm, lifelineToMessages
					.get(l).get(0));

			if(DEBUG) System.out.println("LIFELINE "+l.getName());

			boolean found = searchPaths(startStates, lifelineToMessages.get(l));

			if (!found) {
				 evalLogger.log(Level.WARNING, "Lifeline {0} is INVALID.",
				 l.getName());
				 if(DEBUG) System.out.println(l.getName() + " INVALID");
				valid = false;
			}
		}

		resource.unload();

		return valid;

	}

	private EList<State> getStartStates(Region sm, Message m) {
		// System.out.println("Message "+ m.getBody().getName());

		EList<State> states = new BasicEList<State>();

		for (State state : sm.getStates()) {

			for (Transition t : state.getIncoming()) {

				if (t.getSource() != null) {

					if (t.getTrigger().equals(m.getBody())) {
						states.add(state);
						// System.out.println("Adding start state "+
						// state.getName());
					}
				}
			}

		}
		return states;

	}

	private boolean searchPaths(EList<State> startStates,
			EList<Message> messages) {
		
		for (Message m : messages) {
			if(DEBUG) System.out.println(m.getBody().getName()+"\tSND:\t"+m.getSender().getName()+"\tRCV:\t"+m.getReceiver().getName());
		}


		for (State start : startStates) {

			if(DEBUG) System.out.println("Start state "+ start.getName());

			boolean found = searchPath(start, messages);
			if (found)
				return true;
		}
		
		if(DEBUG)  System.out.println("\t No path found - INVALID!");


		return false;
	}

	private boolean searchPath(State start, EList<Message> messages) {

		int index = 1;
		State currState = start;

		while (index < messages.size()) {

			Message currMessage = messages.get(index);
			if(DEBUG) System.out.println("\tCurr msg "+
			 currMessage.getBody().getName());

			currState = getNextState(currState, currMessage);

			if (currState == null) {
				if(DEBUG) System.out.println("\t Curr state NULL");

				return false;
			} else {
				if(DEBUG)  System.out.println("\tCurr state "+ currState.getName());

				index++;
			}

		}

		return true;
	}

	private State getNextState(State state, Message m) {

		State next = null;

		for (Transition outTrans : state.getOutgoing()) {

			if (outTrans.getTrigger().equals(m.getBody())) {
				next = outTrans.getTarget();
			}
		}

		return next;
	}

	/*
	 * FIXME Maybe this should be somewhere else
	 */
	private HashMap<Lifeline, EList<Message>> getMap(MultiviewModel mmmodel) {

		EList<Lifeline> lifelines = mmmodel.getSequenceview().getLifelines();

		HashMap<Lifeline, EList<Message>> lifelineToMessages = new HashMap<Lifeline, EList<Message>>();

		for (Lifeline l : lifelines) {

			if (l != null) {
				for (LifelineElement le : l.getElements()) {
					if (le instanceof ReceiveEvent) {
						if (!lifelineToMessages.containsKey(l)) {
							lifelineToMessages
									.put(l, new BasicEList<Message>());
						}
						lifelineToMessages.get(l).add(((ReceiveEvent)le).getMessage());
					}
				}

			}
		}

		return lifelineToMessages;
	}

	/**
	 * @param args
	 * @throws IOException
	 * @throws MatchException
	 */
	public static void main(String[] args) {

		if (args.length > 0) {
			// init metamodel
			EPackage.Registry.INSTANCE.put(
					MultiviewPackage.eINSTANCE.getNsURI(),
					MultiviewPackage.eINSTANCE);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
					FILE_EXTENSION, new UUIDResourceFactoryImpl());
		}
		if (args.length == 1 && new File(args[0]).isDirectory()) {
			// run ModelVerifier recursively on all folders in SOURCE_DIR,
			// including SOURCE_DIR
			runModelVerifierRecursively(new File(args[0]));
			// logger.log(Level.INFO, "{0} instances performed.", counter);
		} else if (args.length == 1
				&& args[0].substring(args[0].lastIndexOf(".") + 1,
						args[0].length()).equals("mvml")) {
			ModelVerifier verifier = new ModelVerifier();
			
			boolean valid = verifier.verify(new File(args[0]));
			
			if(valid) System.out.println("Model "+args[0]+" is VALID");
			else System.out.println("Model "+args[0]+" is INVALID");
			
		} else {
			System.out.println("SYNOPSIS \n"
					+ "ModelVerifier SOURCE_FILE/DIR \n\n" + "DESCRIPTION \n"
					+ "Verifies all .mvml models in SOURCE_DIR. \n\n");
		}
	}

	/**
	 * Runs an instance of {@link SATEncodingTester} for each testFolder within
	 * the given file.
	 * 
	 * @param file
	 *            The file to start the recursive execution.
	 */
	private static void runModelVerifierRecursively(File file) {
		if (file.isDirectory()) {
			verifyDir(file);
			File[] files = file.listFiles();
			for (File f : files) {
				if (f.isDirectory()) {
					runModelVerifierRecursively(f);
				}
			}
		}
	}

	private static void verifyDir(File path) {

		for (File file : path.listFiles()) {

			if (file.getName()
					.substring(file.getName().lastIndexOf(".") + 1,
							file.getName().length()).equals("mvml")) {
				ModelVerifier verifier = new ModelVerifier();
				
				boolean valid = verifier.verify(file);
				if(valid) System.out.println("Model "+file.getName()+" is VALID");
				else System.out.println("Model "+file.getName()+" is INVALID");
			}

		}

	}
}
