/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */
package org.modelevolution.multiview.diff.encoding.test;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Scanner;
import java.util.StringTokenizer;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.compare.diff.metamodel.ComparisonResourceSnapshot;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.modelevolution.multiview.Message;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.MultiviewPackage;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.conflictreport.ConflictReport;
import org.modelevolution.multiview.conflictreport.ConflictReportFactory;
import org.modelevolution.multiview.diff.encoding.engine.ICnfParser;
import org.modelevolution.multiview.diff.encoding.engine.IEncodingEngine;
import org.modelevolution.multiview.diff.encoding.engine.ILogObservable;
import org.modelevolution.multiview.diff.encoding.engine.impl.Sequence2SATDIMACSEncodingEngine;
import org.modelevolution.multiview.diff.encoding.engine.impl.Sequence2SATEncodingEngine;
import org.modelevolution.multiview.diff.encoding.service.EncodingService;
import org.modelevolution.multiview.merge.MergeAdvice;
import org.modelevolution.multiview.merge.engine.impl.MultiviewMergeEngine;
import org.modelevolution.multiview.merge.service.MergeService;
import org.modelversioning.conflicts.detection.impl.ThreeWayDiffProvider;
import org.modelversioning.core.diff.service.DiffService;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;
import org.modelversioning.core.match.MatchException;
import org.modelversioning.core.match.engine.impl.DefaultMatchEngineSelector;
import org.modelversioning.core.match.service.MatchService;
import org.modelversioning.core.util.UUIDUtil;

//import com.sun.org.apache.bcel.internal.util.Class2HTML;

import at.jku.fmv.qbf.nnf.parser.ParserBooleSAT;

/**
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * @author <a href="mailto:widl@big.tuwien.ac.at">Magdalena Widl</a>
 * 
 */
public class SATEncodingRunner implements ILogObservable {
	/**
	 * logger
	 */
	private static final Logger logger = Logger
			.getLogger("org.modelevolution.multiview");
	private static Handler loggerHandler;

	private static final Logger evalLogger = Logger
			.getLogger("org.modelevolution.multiview.eval");
	private static Handler evalLoggerHandler;

	private static final Logger timeLogger = Logger
			.getLogger("org.modelevolution.multiview.eval.time");
	private static Handler timeLoggerHandler;

	/**
	 * static properties
	 */
	public static String MODEL_NAME = "model";
	public static final String MODEL_NAME_REGEX = MODEL_NAME + "\\d+";
	public static String OUTPUT_FILE = "encoding.sat";
	public static String CNF_FILE = "out.cnf";
	public static String SOLUTION_FILE = "solution";
	/**
	 * BEGIN DO NOT EDIT
	 * 
	 * Customize the following values only via the file SDMerger.properties
	 * which may be found in the root directory of the plugin
	 * org.modelevolution.multiview.diff.encoding.test
	 */
	public static final String MVML_FILE_EXTENSION = "mvml";
	public static final String MVCR_FILE_EXTENSION = "mvcr";
	public static String ORIGIN_FILE = "origin";
	public static String WCP1_FILE = "working_copy_1";
	public static String WCP2_FILE = "working_copy_2";
	public static String CR_FILE = "conflictreport";
	public static String MERGED_MODEL_NAME = "merged";
	public static long MAX_MODELS = Long.MAX_VALUE;
	public static String PICOSAT_PATH = new File("models/picosat-936/picosat")
			.getAbsolutePath();
	public static String TSEITIN_PATH = new File("models/nnf.jar")
			.getAbsolutePath();
	@SuppressWarnings("rawtypes")
	public static Class encodingEngineClass = Sequence2SATDIMACSEncodingEngine.class;
	/**
	 * END DO NOT EDIT
	 */

	/**
	 * class variables
	 */
	private ResourceSet resourceSet = null;
	private Resource origin = null;
	private Resource workingCopy1 = null;
	private Resource workingCopy2 = null;
	private Resource mergedModelResource = null;
	private Resource conflictModelResource = null;

	private ComparisonResourceSnapshot leftCRS2 = null;
	private ComparisonResourceSnapshot rightCRS2 = null;

	private ThreeWayDiffProvider threeWayDiff = null;
	private ConflictReport conflictReport = null;

	private EncodingService encodingService = null;
	private MergeService mergeService = null;

	private ModelVerifier verifier = null;

	private ParserBooleSAT parser = null;

	private long startTime = 0;
	private long currentModelTime = 0;
	private long totalTime = 0;
	private static int testInstanceCounter = 0;
	private long nrModels = 0;
	private long nrInvalidModels = 0;

	static {
		Properties prop = new Properties();
		try {
			prop.load(new FileInputStream("SDMerger.properties"));

			if (prop.getProperty("path.picosat") != null) {
				PICOSAT_PATH = new File(prop.getProperty("path.picosat"))
						.getAbsolutePath();
			}
			if (prop.getProperty("path.tseitin") != null) {
				TSEITIN_PATH = new File(prop.getProperty("path.tseitin"))
						.getAbsolutePath();
			}
			if (prop.getProperty("name.origin") != null) {
				ORIGIN_FILE = prop.getProperty("name.origin");
			}
			if (prop.getProperty("name.working_copy1") != null) {
				WCP1_FILE = prop.getProperty("name.working_copy1");
			}
			if (prop.getProperty("name.working_copy2") != null) {
				WCP2_FILE = prop.getProperty("name.working_copy2");
			}
			if (prop.getProperty("name.conflictreport") != null) {
				CR_FILE = prop.getProperty("name.conflictreport");
			}
			if (prop.getProperty("name.merged") != null) {
				MERGED_MODEL_NAME = prop.getProperty("name.merged");
			}
			if (prop.getProperty("value.max_models") != null) {
				if (prop.getProperty("value.max_models").equals(
						"Long.MAX_VALUE")) {
					MAX_MODELS = Long.MAX_VALUE;
				} else if (prop.getProperty("value.max_models").equals(
						"Long.MIN_VALUE")) {
					MAX_MODELS = Long.MIN_VALUE;
				} else {
					try {
						MAX_MODELS = Long.parseLong(prop
								.getProperty("value.max_models"));
					} catch (NumberFormatException e) {
						logger.log(
								Level.WARNING,
								"Unexpected value of property max_models: {0}. "
										+ "Falling back to default value: {1}",
								new Object[] {
										prop.getProperty("value.max_models"),
										MAX_MODELS });
					}
				}
			}
			if (prop.getProperty("class.IEncodingEngine") != null) {
				ClassLoader classLoader = SATEncodingRunner.class
						.getClassLoader();

				Class tmpEncodingEngineClass = null;
				try {
					tmpEncodingEngineClass = classLoader.loadClass(prop
							.getProperty("class.IEncodingEngine"));
				} catch (ClassNotFoundException e) {
					logger.log(Level.SEVERE, e.getMessage());
				}
				if (tmpEncodingEngineClass != null) {
					encodingEngineClass = tmpEncodingEngineClass;
				}
			}

			// prop.setProperty("path.picosat", PICOSAT_PATH);
			// prop.setProperty("path.tseitin", TSEITIN_PATH);
			// prop.setProperty("path.util", UTIL_PATH);
			// prop.setProperty("name.origin", ORIGIN_FILE);
			// prop.setProperty("name.working_copy1", WCP1_FILE);
			// prop.setProperty("name.working_copy2", WCP2_FILE);
			// prop.setProperty("name.conflictreport", CR_FILE);
			// prop.setProperty("name.merged", MERGED_MODEL_NAME);
			// prop.setProperty("value.max_models", new
			// Long(MAX_MODELS).toString());

			// prop.store(new FileOutputStream("SDMerger.properties"),
			// "SDMerger configuration");
		} catch (IOException e) {
			logger.log(Level.WARNING, e.getMessage());
		}
	}

	public SATEncodingRunner() {
		logger.log(Level.INFO, "Starting {0}", this.getClass().getName());

		IEncodingEngine encodingEngine = null;
		
		try {
			encodingEngine = (IEncodingEngine)Class.forName(encodingEngineClass.getCanonicalName()).newInstance();
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}

		if (encodingEngine == null) {
			logger.log(Level.SEVERE,
					"IEncodingEngine could not be instantiated.");
			System.exit(1);
		}

		encodingService = new EncodingService(encodingEngine);
		mergeService = new MergeService(new MultiviewMergeEngine());
		verifier = new ModelVerifier();
	}

	public void performMerge(String path) {
		logger.log(Level.INFO, "Starting {0} with path {1}.",
				new Object[] {
						encodingService.getEncodingEngine().getClass()
								.getName(), path });

		if (!path.endsWith(File.separator)) {
			path += File.separator;
		}

		try {
			setUp(path);

			// generate SAT encoding
			if (encodingService.getEncodingEngine() instanceof ICnfParser) {
				File encodingOutput = new File(path + CNF_FILE);
				encodingService.generateEncodedDiffModel(threeWayDiff,
						conflictReport, encodingOutput);
			} else {
				File encodingOutput = new File(path + OUTPUT_FILE);
				encodingService.generateEncodedDiffModel(threeWayDiff,
						conflictReport, encodingOutput);
				parser = new ParserBooleSAT(100000, encodingOutput);
				parser.toCNF(path + CNF_FILE);
			}

			conflictModelResource.save(Collections.EMPTY_MAP);

			if (encodingService.getEncodingEngine() instanceof ILogObservable) {
				logObservations.put(LogObservation.ENC,
						((ILogObservable) encodingService.getEncodingEngine())
								.getLogObservation(LogObservation.ENC));
				logObservations.put(LogObservation.DIFF_POST,
						((ILogObservable) encodingService.getEncodingEngine())
								.getLogObservation(LogObservation.DIFF_POST));
				logObservations
						.put(LogObservation.CNF_VAR_COUNT,
								((ILogObservable) encodingService
										.getEncodingEngine())
										.getLogObservation(LogObservation.CNF_VAR_COUNT));
				logObservations
						.put(LogObservation.CNF_CLAUSE_COUNT,
								((ILogObservable) encodingService
										.getEncodingEngine())
										.getLogObservation(LogObservation.CNF_CLAUSE_COUNT));
			}

			// generate merged models
			getModels(path);

			testInstanceCounter++;

		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		} catch (MatchException e) {
			logger.log(Level.SEVERE, e.getMessage());
		} finally {
			tearDown();
		}
		logger.log(Level.INFO, "{0} with path {1} done.", new Object[] {
				this.getClass().getName(), path });

		// *LOG EVAL RESULTS*********************************************
		evalLogger.log(Level.INFO, "Test instance: {0}\n"
				+ "Solutions found: {1}\n" + "Invalid solutions: {2}\n"
				+ "Origin Msg Count: {3}\nAdded Msg Count: {4}\n"
				+ "IEncodingEngine: {5}\n", new Object[] { path, nrModels,
				nrInvalidModels, getLogObservation(LogObservation.MSG_COUNT),
				getLogObservation(LogObservation.ADD_MSG_COUNT),
				encodingService.getEncodingEngine().getClass().getName() });
		// **************************************************************
	}

	private void getModels(String path) {
		logger.log(Level.INFO, "Starting SAT solver.");

		File cnfFile = new File(path + CNF_FILE);
		nrModels = 0;
		nrInvalidModels = 0;

		startTime = System.currentTimeMillis();
		File solution = runSATsolver(cnfFile, path);
		logObservations.put(LogObservation.SAT, System.currentTimeMillis()
				- startTime);

		while (isSAT(solution) && nrModels < MAX_MODELS) {
			logger.log(Level.INFO,
					"Solution found. Starting variable extraction.");

			// decode sat model
			startTime = System.currentTimeMillis();
			nrModels++;
			List<Integer> vars = extractVars(solution);

			logger.log(Level.INFO, "Starting mergeAdvice extraction.");
			MergeAdvice mergeAdvice = getMergeAdvice(vars);

			logObservations.put(LogObservation.DEC, System.currentTimeMillis()
					- startTime);

			// merge model
			logger.log(Level.INFO, "Starting model merger.");
			startTime = System.currentTimeMillis();
			File model = buildModel(mergeAdvice, nrModels, path);
			logObservations.put(LogObservation.MERGE,
					System.currentTimeMillis() - startTime);

			// verify merged model
			logger.log(Level.INFO, "Starting verifier.");
			startTime = System.currentTimeMillis();
			if (!verifier.verify(model)) {
				evalLogger.log(Level.SEVERE, "Model {0} is INVALID!",
						model.getAbsolutePath());
				nrInvalidModels++;
				logObservations.put(LogObservation.VALID, 0l);
				 System.exit(1);
			} else {
				logObservations.put(LogObservation.VALID, 1l);
			}
			logObservations.put(LogObservation.VERIFY,
					System.currentTimeMillis() - startTime);

			// done. log execution time
			logTimeObservation();

			// start next run. update encoding
			logger.log(Level.INFO, "Starting addToCNF.");
			startTime = System.currentTimeMillis();
			cnfFile = addToCNF(vars, cnfFile);
			logger.log(Level.INFO, "Starting increaseClauses.");
			cnfFile = increaseNrClauses(cnfFile, 1);
			logObservations.put(LogObservation.ENC, System.currentTimeMillis()
					- startTime);

			// run sat solver
			logger.log(Level.INFO, "Starting SAT solver.");
			startTime = System.currentTimeMillis();
			solution = runSATsolver(cnfFile, path);
			logObservations.put(LogObservation.SAT, System.currentTimeMillis()
					- startTime);
		}
		if (!isSAT(solution)) {
			// no more models found. log execution time
			logger.log(Level.INFO, "No solution found.");
			logTimeObservation(-1);
		}
	}

	private File addToCNF(List<Integer> vars, File cnfFile) {

		try {
			BufferedWriter cnf = new BufferedWriter(new FileWriter(cnfFile,
					true));

			for (int var : vars) {
				String originVar = null;

				if (encodingService.getEncodingEngine() instanceof ICnfParser) {
					originVar = ((ICnfParser) encodingService
							.getEncodingEngine()).getVariable(var);

					if (originVar != null
							&& originVar
									.endsWith(Sequence2SATDIMACSEncodingEngine.VAR_EXT_MESSAGE)) {
						cnf.write("-" + var + " ");
					}
				} else {
					originVar = parser.getVariable(var);

					if (originVar != null
							&& originVar
									.endsWith(Sequence2SATEncodingEngine.VAR_EXT_MESSAGE)) {
						cnf.write("-" + var + " ");
					}
				}
			}
			cnf.write("0\n");
			cnf.close();
		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		return cnfFile;
	}

	private MergeAdvice getMergeAdvice(List<Integer> vars) {

		MergeAdvice mergeAdvice = new MergeAdvice();

		for (int var : vars) {
			String originVar = null;

			if (encodingService.getEncodingEngine() instanceof ICnfParser) {
				originVar = ((ICnfParser) encodingService.getEncodingEngine())
						.getVariable(var);

				if (originVar != null
						&& originVar
								.endsWith(Sequence2SATDIMACSEncodingEngine.VAR_EXT_MESSAGE)) {
					mergeAdvice.addPosition(((ICnfParser) encodingService
							.getEncodingEngine()).getVariable(var));
				}
			} else {
				originVar = parser.getVariable(var);

				if (originVar != null
						&& originVar
								.endsWith(Sequence2SATEncodingEngine.VAR_EXT_MESSAGE)) {
					mergeAdvice.addPosition(parser.getVariable(var));
				}
			}
		}
		return mergeAdvice;
	}

	private File buildModel(MergeAdvice mergeAdvice, long count, String path) {

		File model = new File(path + MERGED_MODEL_NAME + count + "."
				+ MVML_FILE_EXTENSION);

		// generate merged model(s)
		try {
			mergedModelResource = resourceSet.createResource(URI
					.createFileURI(model.getAbsolutePath()));

			mergeService.generateMergedModel(threeWayDiff, mergeAdvice,
					mergedModelResource, conflictModelResource);

			mergedModelResource.save(Collections.EMPTY_MAP);
		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}
		mergedModelResource.unload();

		return model;

	}

	private File increaseNrClauses(File cnf, int nrIncrease) {
		File newFile = new File(cnf.getAbsolutePath() + ".tmp");

		try {
			BufferedReader reader = new BufferedReader(new FileReader(cnf));
			PrintWriter writer = new PrintWriter(new FileWriter(newFile));
			String line = null;

			while ((line = reader.readLine()) != null) {

				if (line.startsWith("p")) {

					StringTokenizer tokenizer = new StringTokenizer(line, " ");
					writer.print(tokenizer.nextToken() + " ");
					writer.print(tokenizer.nextToken() + " ");
					writer.print(tokenizer.nextToken() + " ");

					int nrClauses = Integer.parseInt(tokenizer.nextToken())
							+ nrIncrease;
					writer.print(nrClauses + " ");
					writer.println();

					logObservations.put(LogObservation.CNF_CLAUSE_COUNT,
							new Long(nrClauses));
				} else {
					writer.println(line);
				}
			}
			reader.close();
			writer.close();
			moveFile(newFile, cnf);

		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		return cnf;
	}

	private List<Integer> extractVars(File solution) {

		List<Integer> vars = new ArrayList<Integer>();

		try {
			Scanner scanner = new Scanner(solution);
			scanner.nextLine();

			while (scanner.hasNext()) {

				String currVar = scanner.next();

				if (!currVar.startsWith("-") && !currVar.startsWith("v")) {

					vars.add(Integer.parseInt(currVar));
				}
			}

		} catch (FileNotFoundException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		return vars;
	}

	private boolean isSAT(File solution) {

		try {
			Scanner scanner = new Scanner(solution);
			scanner.skip("s");
			String result = scanner.next();

			if (result.equals("SATISFIABLE")) {
				return true;
			}

		} catch (FileNotFoundException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		return false;
	}

	private File runSATsolver(File cnf, String path) {

		String cnfPath = cnf.getAbsolutePath();

		try {
			Runtime rt = Runtime.getRuntime();
			Process p;

			p = rt.exec(new String[] { PICOSAT_PATH, cnfPath });

			BufferedReader reader = new BufferedReader(new InputStreamReader(
					p.getInputStream()));
			String line;

			FileWriter fstream = new FileWriter(path + File.separator
					+ SOLUTION_FILE);
			BufferedWriter out = new BufferedWriter(fstream);

			while (reader.ready() == false) { /* intentional empty space here */
			}
			while ((line = reader.readLine()) != null) {
				out.write(line + "\n");
			}
			p.waitFor();
			out.close();

		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		} catch (InterruptedException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		return new File(path + File.separator + SOLUTION_FILE);
	}

	/**
	 * Helper method for setting up resources.
	 * 
	 * @param path
	 * @throws MatchException
	 */
	private void setUp(String path) throws MatchException {
		// load resources
		resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(MVML_FILE_EXTENSION, new UUIDResourceFactoryImpl());
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(MVCR_FILE_EXTENSION, new UUIDResourceFactoryImpl());
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put("xmi", new UUIDResourceFactoryImpl());

		URI fileURI_id = URI.createFileURI(new File(path + ORIGIN_FILE + "."
				+ MVML_FILE_EXTENSION).getAbsolutePath());

		origin = resourceSet.getResource(fileURI_id, true);
		URI fileURI1_id = URI.createFileURI(new File(path + WCP1_FILE + "."
				+ MVML_FILE_EXTENSION).getAbsolutePath());
		workingCopy1 = resourceSet.getResource(fileURI1_id, true);
		UUIDUtil.addUUIDs(workingCopy1.getContents().get(0));

		URI fileURI2_id = URI.createFileURI(new File(path + WCP2_FILE + "."
				+ MVML_FILE_EXTENSION).getAbsolutePath());
		workingCopy2 = resourceSet.getResource(fileURI2_id, true);
		UUIDUtil.addUUIDs(workingCopy2.getContents().get(0));

		URI fileURI_cr = URI.createFileURI(new File(path + CR_FILE + "."
				+ MVCR_FILE_EXTENSION).getAbsolutePath());
		conflictModelResource = resourceSet.createResource(fileURI_cr);

		try {
			this.workingCopy1.save(Collections.EMPTY_MAP);
			this.workingCopy2.save(Collections.EMPTY_MAP);
		} catch (IOException e) {
			logger.log(Level.SEVERE, e.getMessage());
		}

		// match and diff models
		startTime = System.currentTimeMillis();
		MatchService matchService = new MatchService(
				new DefaultMatchEngineSelector(false));
		DiffService diffService = new DiffService();

		leftCRS2 = diffService.generateComparisonResourceSnapshot(matchService
				.generateMatchModel(origin, workingCopy1));
		rightCRS2 = diffService.generateComparisonResourceSnapshot(matchService
				.generateMatchModel(origin, workingCopy2));
		// create ThreeWayDiffProvider instance
		threeWayDiff = new ThreeWayDiffProvider(leftCRS2, rightCRS2);
		logObservations.put(LogObservation.DIFF_EMF, System.currentTimeMillis()
				- startTime);

		conflictReport = ConflictReportFactory.eINSTANCE.createConflictReport();
		conflictReport.setLeftComparisonSnapshot(leftCRS2);
		conflictReport.setRightComparisonSnapshot(rightCRS2);
		conflictModelResource.getContents().add(conflictReport);

		logObservations.put(LogObservation.MSG_COUNT, new Long(
				((MultiviewModel) origin.getContents().get(0))
						.getSequenceview().getMessages().size()));

		long nrAddedMessages = 0;
		for (EObject o : threeWayDiff.getAddedEObjects(Side.LEFT, true)) {
			if (o instanceof Message) {
				nrAddedMessages += 1;
			}
		}
		for (EObject o : threeWayDiff.getAddedEObjects(Side.RIGHT, true)) {
			if (o instanceof Message) {
				nrAddedMessages += 1;
			}
		}
		logObservations.put(LogObservation.ADD_MSG_COUNT, nrAddedMessages);

		long nrStates = 0;
		long nrTransitions = 0;

		for (Region r : ((MultiviewModel) origin.getContents().get(0))
				.getStateview().getStatemachines()) {
			nrStates += r.getStates().size();

			for (State s : r.getStates()) {
				nrTransitions += s.getIncoming().size();
			}
		}
		logObservations.put(LogObservation.STATE_COUNT, nrStates);
		logObservations.put(LogObservation.TRANSITION_COUNT, nrTransitions);
	}

	/**
	 * Helper method for closing all resources.
	 */
	private void tearDown() {
		try {
			conflictModelResource.save(Collections.EMPTY_MAP);
			this.conflictModelResource.unload();
			this.origin.unload();
			this.workingCopy1.unload();
			this.workingCopy2.unload();
		} catch (Exception e) {
			logger.log(Level.SEVERE, e.getMessage());
		}
	}

	/**
	 * @param args
	 * @throws SecurityException
	 * @throws IOException
	 * @throws MatchException
	 */
	public static void main(String[] args) throws SecurityException,
			IOException {
		// configure logger
		String logname = "";
		if (args.length == 1) {
			logname = args[0];
		} else if (args.length == 2) {
			logname = args[1];
		}

		loggerHandler = new FileHandler(logname + File.separator + new Date()
				+ ".log", true);
		loggerHandler.setFormatter(new SimpleFormatter());
		logger.addHandler(loggerHandler);
		logger.setUseParentHandlers(false);
		logger.setLevel(Level.INFO);

		evalLoggerHandler = new FileHandler(logname + File.separator
				+ new Date() + "_eval.log", true);
		evalLoggerHandler.setFormatter(new SimpleFormatter());
		evalLogger.addHandler(evalLoggerHandler);
		evalLogger.addHandler(new ConsoleHandler());
		evalLogger.setLevel(Level.INFO);

		if (args.length > 0) {
			// init metamodel
			EPackage.Registry.INSTANCE.put(
					MultiviewPackage.eINSTANCE.getNsURI(),
					MultiviewPackage.eINSTANCE);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
					MVML_FILE_EXTENSION, new UUIDResourceFactoryImpl());
		}
		if (args.length == 1 && isTestFolder(new File(args[0]))) {

			// run SATEncodingTester on SOURCE_DIR
			File srcDir = new File(args[0]);
			SATEncodingRunner tester = new SATEncodingRunner();
			tester.performMerge(srcDir.getPath());
		} else if (args.length == 2 && args[0].equals("-r")
				&& new File(args[1]).isDirectory()) {
			// run SATEncodingTester recursively on all folders in SOURCE_DIR
			runSATEncodingTesterRecursively(new File(args[1]));

		} else {
			System.out
					.println("SYNOPSIS \n"
							+ "SATEncodingTester [OPTION] SOURCE_DIR \n\n"
							+ "DESCRIPTION \n"
							+ "Encodes the Merge Problem of the given SOURCE_DIR in SAT "
							+ "and generates possible merged models. \n\n"
							+ "-r \t follow directory tree recursively. \n");
		}

		evalLogger.log(Level.INFO, "Done. {0} instances performed.",
				testInstanceCounter);

		loggerHandler.flush();
		loggerHandler.close();
		evalLoggerHandler.flush();
		evalLoggerHandler.close();
		timeLoggerHandler.flush();
		timeLoggerHandler.close();
	}

	/**
	 * @param path
	 * @throws IOException
	 */
	private static void initTimeLogger(String path) throws IOException {
		for (Handler h : timeLogger.getHandlers()) {
			h.flush();
			h.close();
			timeLogger.removeHandler(h);
		}

		timeLoggerHandler = new FileHandler(path + File.separator
				+ path.replaceAll(File.separator, "_") + "-" + new Date()
				+ "_time.log", false);
		timeLoggerHandler.setFormatter(new Formatter() {

			@Override
			public String format(LogRecord record) {
				StringBuilder builder = new StringBuilder();
				builder.append(formatMessage(record));
				return builder.toString();
			}
		});
		timeLogger.addHandler(timeLoggerHandler);
		timeLogger.setLevel(Level.INFO);
		timeLogger.setUseParentHandlers(false);

		timeLogger
				.log(Level.INFO,
						"nr\tdiff_emf\tdiff_post\tenc\tsat\tdec\tmerge\tverify\tsum\tssum\tmsg\tmsg_add\tstates\ttransitions\tvar\tclauses\tvalid\n");
	}

	/**
	 * Runs an instance of {@link SATEncodingTester} for each testFolder within
	 * the given file.
	 * 
	 * @param file
	 *            The file to start the recursive execution.
	 */
	private static void runSATEncodingTesterRecursively(File file) {
		if (file.isDirectory()) {
			if (isTestFolder(file)) {
				try {
					initTimeLogger(file.getPath().toString());
				} catch (IOException e) {
					logger.log(Level.SEVERE, e.getMessage());
				}

				SATEncodingRunner tester = new SATEncodingRunner();
				tester.performMerge(file.getPath());
			}
			File[] files = file.listFiles();
			for (File f : files) {
				if (f.isDirectory()) {
					runSATEncodingTesterRecursively(f);
				}
			}
		}
	}

	/**
	 * Checks whether the given file is a directory and contains the expected
	 * files ORIGIN_FILE, WCP1_FILE, and WCP2_FILE.
	 * 
	 * @param file
	 *            The file to check.
	 * @return true, if the given file is a valid test directory; false
	 *         otherwise.
	 */
	private static boolean isTestFolder(File file) {
		if (file.isDirectory()) {
			if (new File(file.getPath() + File.separator + ORIGIN_FILE + "."
					+ MVML_FILE_EXTENSION).exists()
					&& new File(file.getPath() + File.separator + WCP1_FILE
							+ "." + MVML_FILE_EXTENSION).exists()
					&& new File(file.getPath() + File.separator + WCP2_FILE
							+ "." + MVML_FILE_EXTENSION).exists()) {
				return true;
			}
		}
		return false;
	}

	/*
	 * Workaround for File.renameTo(File dest) (didn't work for me)
	 */
	public static void moveFile(File sourceFile, File destFile)
			throws IOException {
		if (!destFile.exists()) {
			destFile.createNewFile();
		}

		FileChannel source = null;
		FileChannel destination = null;
		try {
			source = new FileInputStream(sourceFile).getChannel();
			destination = new FileOutputStream(destFile).getChannel();

			long count = 0;
			long size = source.size();
			while ((count += destination.transferFrom(source, count, size
					- count)) < size)
				;
		} finally {
			if (source != null) {
				source.close();
			}
			if (destination != null) {
				destination.close();
			}
		}
		sourceFile.delete();
	}

	@Override
	public Long getLogObservation(LogObservation observation) {
		if (logObservations != null && logObservations.containsKey(observation) && logObservations.get(observation) != null) {
			return logObservations.get(observation);
		}
		return 0l;
	}

	/**
	 * Logs the time observation of the current run.
	 */
	private void logTimeObservation() {
		logTimeObservation(nrModels);
	}

	private void logTimeObservation(long modelCounter) {
		String emptyVal = "X";
		long emptyTime = -1l;

		currentModelTime = (getLogObservation(LogObservation.DIFF_EMF)
				+ getLogObservation(LogObservation.DIFF_POST)
				+ getLogObservation(LogObservation.ENC)
				+ getLogObservation(LogObservation.SAT)
				+ getLogObservation(LogObservation.DEC)
				+ getLogObservation(LogObservation.MERGE) + getLogObservation(LogObservation.VERIFY));
		totalTime += currentModelTime;

		timeLogger
				.log(Level.INFO,
						"{0}\t{1}\t{2}\t{3}\t{4}\t{5}\t{6}\t{7}\t{8}\t{9}\t{10}\t{11}\t{12}\t{13}\t{14}\t{15}\t{16}\n",
						new Object[] {
								modelCounter != emptyTime ? modelCounter
										: emptyVal,
								getLogObservation(LogObservation.DIFF_EMF) != emptyTime ? getLogObservation(LogObservation.DIFF_EMF)
										: emptyVal,
								getLogObservation(LogObservation.DIFF_POST) != emptyTime ? getLogObservation(LogObservation.DIFF_POST)
										: emptyVal,
								getLogObservation(LogObservation.ENC) != emptyTime ? getLogObservation(LogObservation.ENC)
										: emptyVal,
								getLogObservation(LogObservation.SAT) != emptyTime ? getLogObservation(LogObservation.SAT)
										: emptyVal,
								getLogObservation(LogObservation.DEC) != emptyTime ? getLogObservation(LogObservation.DEC)
										: emptyVal,
								getLogObservation(LogObservation.MERGE) != emptyTime ? getLogObservation(LogObservation.MERGE)
										: emptyVal,
								getLogObservation(LogObservation.VERIFY) != emptyTime ? getLogObservation(LogObservation.VERIFY)
										: emptyVal,
								currentModelTime,
								totalTime,
								getLogObservation(LogObservation.MSG_COUNT),
								getLogObservation(LogObservation.ADD_MSG_COUNT),
								getLogObservation(LogObservation.STATE_COUNT),
								getLogObservation(LogObservation.TRANSITION_COUNT),
								getLogObservation(LogObservation.CNF_VAR_COUNT),
								getLogObservation(LogObservation.CNF_CLAUSE_COUNT),
								getLogObservation(LogObservation.VALID) });

		// clear time observations
		logObservations.put(LogObservation.DIFF_EMF, emptyTime);
		logObservations.put(LogObservation.DIFF_POST, emptyTime);
		logObservations.put(LogObservation.ENC, emptyTime);
		logObservations.put(LogObservation.SAT, emptyTime);
		logObservations.put(LogObservation.DEC, emptyTime);
		logObservations.put(LogObservation.MERGE, emptyTime);
		logObservations.put(LogObservation.VERIFY, emptyTime);
		logObservations.put(LogObservation.VALID, emptyTime);
	}
}