/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */
package org.modelevolution.multiview.diff.encoding.test;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Hashtable;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.modelevolution.multiview.Message;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.MultiviewPackage;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.SequenceView;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.Symbol;
import org.modelevolution.multiview.Transition;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;

/**
 * Reduces the {@link Region}s of a given {@link MultiviewModel} to the
 * {@link Transition}s and {@link State}s used in the {@link SequenceView}.
 * 
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class StateMachineSimplifyer {
	/**
	 * logger
	 */
	private static final Logger logger = Logger
			.getLogger("org.modelevolution.multiview.StateMachineSimplifyer");

	/**
	 * static variables
	 */
	public static final String MVML_FILE_EXTENSION = "mvml";
	public static final String SIMPLE_FILE_POSTFIX = "_simple";

	/**
	 * class variables
	 */
	private ResourceSet resourceSet = null;
	private Resource originModel = null;
	private Resource simpleModel = null;
	private MultiviewModel mvmlModel = null;
	private Hashtable<String, EList<Message>> symbolMessageTable;

	public StateMachineSimplifyer(String modelPath) {
		// init metamodel
		EPackage.Registry.INSTANCE.put(MultiviewPackage.eINSTANCE.getNsURI(),
				MultiviewPackage.eINSTANCE);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				MVML_FILE_EXTENSION, new UUIDResourceFactoryImpl());

		// load resources
		resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(MVML_FILE_EXTENSION, new UUIDResourceFactoryImpl());
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put("xmi", new UUIDResourceFactoryImpl());

		URI originModelURI = URI.createFileURI(new File(modelPath)
				.getAbsolutePath());
		originModel = resourceSet.getResource(originModelURI, true);

		URI simpleModelURI = URI.createFileURI(new File(modelPath.replaceFirst(
				"." + MVML_FILE_EXTENSION, SIMPLE_FILE_POSTFIX + "."
						+ MVML_FILE_EXTENSION)).getAbsolutePath());
		simpleModel = resourceSet.createResource(simpleModelURI);

		if (originModel.getContents().get(0) instanceof MultiviewModel) {
			mvmlModel = (MultiviewModel) originModel.getContents().get(0);

			EList<Region> removeRegions = new BasicEList<Region>();
			EList<State> removeStates = new BasicEList<State>();

			for (Region r : mvmlModel.getStateview().getStatemachines()) {
				if (r.getClass_().getLifelines() == null
						|| r.getClass_().getLifelines().isEmpty()) {
					logger.log(Level.INFO, "Region {0} unused.", r.getName());
					removeRegions.add(r);
				} else {
					Transition initialTransition = r.getInitialTransition();
					if (initialTransition != null
							&& !hasMessage(initialTransition.getTrigger())) {
						logger.log(
								Level.INFO,
								"Transition {0}-{1}-{2} unused.",
								new Object[] {
										initialTransition.getSource().getName(),
										initialTransition.getTrigger()
												.getName(),
										initialTransition.getTarget().getName() });

						r.setInitialTransition(null);
						initialTransition.getTarget().getIncoming()
								.remove(initialTransition);
						initialTransition = null;
					}

					for (State s : r.getStates()) {
						EList<Transition> removeTransitions = new BasicEList<Transition>();
						for (Transition t : s.getOutgoing()) {
							if (!hasMessage(t.getTrigger())) {
								logger.log(Level.INFO,
										"Transition {0}-{1}-{2} unused.",
										new Object[] { t.getSource().getName(),
												t.getTrigger().getName(),
												t.getTarget().getName() });
								removeTransitions.add(t);
							}
						}
						s.getOutgoing().removeAll(removeTransitions);
						for (Transition rt : removeTransitions) {
							if (rt.getTarget() != null
									&& rt.getTarget().getIncoming() != null) {
								rt.getTarget().getIncoming().remove(rt);
							}
						}
						if ((s.getOutgoing() == null || s.getOutgoing()
								.isEmpty())
								&& s.getIncoming() == null
								|| s.getIncoming().isEmpty()) {
							removeStates.add(s);
						}
					}
					r.getStates().removeAll(removeStates);
				}
			}
			mvmlModel.getStateview().getStatemachines()
					.removeAll(removeRegions);
			for (Region rr : removeRegions) {
				rr.getClass_().setStatemachine(null);
			}

			simpleModel.getContents().add(mvmlModel);
			try {
				simpleModel.save(Collections.EMPTY_MAP);
			} catch (IOException e) {
				logger.log(Level.WARNING, e.getMessage());
			}

		} else {
			logger.log(Level.WARNING, "No valid MVML originModel");
		}
	}

	private boolean hasMessage(Symbol symbol) {
		String symbolName = symbol.getName();

		if (symbolMessageTable == null) {
			symbolMessageTable = new Hashtable<String, EList<Message>>();

			for (Message m : mvmlModel.getSequenceview().getMessages()) {
				String s = m.getBody().getName();

				if (symbolMessageTable.contains(s)) {
					EList<Message> msg = symbolMessageTable.get(s);
					msg.add(m);
					symbolMessageTable.put(s, msg);
				} else {
					EList<Message> msg = new BasicEList<Message>();
					msg.add(m);
					symbolMessageTable.put(s, msg);
				}
			}
		}

		boolean hasMessage = symbolMessageTable.containsKey(symbolName);
		return hasMessage;
	}

	public static void main(String[] args) {
		if (args.length > 0) {
			logger.addHandler(new ConsoleHandler());
			logger.setLevel(Level.SEVERE);

			new StateMachineSimplifyer(args[0]);

		} else {
			System.out.println("SYNOPSIS \n"
					+ "StateMachineSimplifyer PATH/TO/MODEL."
					+ MVML_FILE_EXTENSION);
		}
	}
}
