/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */
package org.modelevolution.multiview.diff.encoding.engine.impl;

import java.io.File;
import java.io.IOException;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.modelevolution.multiview.Lifeline;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.SequenceView;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.conflictreport.ConflictReport;
import org.modelevolution.multiview.diff.encoding.engine.IEncodingEngine;
import org.modelversioning.conflictreport.conflict.Conflict;
import org.modelversioning.conflicts.detection.impl.ThreeWayDiffProvider;

/**
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public abstract class SequenceEncodingEngine implements IEncodingEngine {

	/**
	 * The conflictReport to enrich
	 */
	protected ConflictReport conflictReport;

	/**
	 * The union of all lifelines in the left and the right model
	 */
	protected EList<Lifeline> lifelines;

	/**
	 * The set of statemachines
	 */
	protected EList<Region> statemachines;

	/**
	 * The set of states
	 */
	protected EList<State> states;

	@Override
	public void generateEncoding(EList<Conflict> conflicts,
			ThreeWayDiffProvider threeWayDiff, File output) throws IOException {
		initLifelines(threeWayDiff);
		initStatemachines(threeWayDiff);
		initStates(threeWayDiff);
	}
	
	@Override
	public void generateEncoding(ThreeWayDiffProvider threeWayDiff, File output) throws IOException {
		generateEncoding(null, threeWayDiff, output);
	}
	
	@Override
	public void generateEncoding(ThreeWayDiffProvider threeWayDiff,
			ConflictReport conflictReport, File output) throws IOException {
		this.conflictReport = conflictReport;
		generateEncoding(threeWayDiff, output);
	}

	private void initStatemachines(ThreeWayDiffProvider threeWayDiff) {
		statemachines = new BasicEList<Region>();

		for (Lifeline lifeline : lifelines) {
			// create default state machine if there is no predefined one
			if (lifeline.getClass_() == null
					|| lifeline.getClass_().getStatemachine() == null) {
				lifeline.initDummyStatemachine();
			}
			statemachines.add(lifeline.getClass_().getStatemachine());
		}
	}

	private void initStates(ThreeWayDiffProvider threeWayDiff) {
		states = new BasicEList<State>();

		for (Region statemachine : statemachines) {
			states.addAll(statemachine.getStates());
		}
	}

	/**
	 * Calculates the union of {@link Lifeline} <code>lifelines</code> in the
	 * left and the right {@link SequenceView}
	 * 
	 * @param threeWayDiff
	 */
	private void initLifelines(ThreeWayDiffProvider threeWayDiff) {
		lifelines = new BasicEList<Lifeline>();

		SequenceView leftSequenceView = ((MultiviewModel) threeWayDiff
				.getLeftModel().get(0)).getSequenceview();
		lifelines.addAll(leftSequenceView.getLifelines());

		SequenceView rightSequenceView = ((MultiviewModel) threeWayDiff
				.getRightModel().get(0)).getSequenceview();
		for (Lifeline lifeline : rightSequenceView.getLifelines()) {
			if (threeWayDiff.getMatchingEObject(lifeline, Side.RIGHT, false) == null) {
				lifelines.add(lifeline);
			}
		}
	}

	/**
	 * Creates a text representation of the given {@link EObject}s.
	 * 
	 * @param elements
	 * @param featureName
	 * @param delimiter
	 * @return a flattened string
	 */
	protected String flatten(EList<? extends EObject> elements,
			String featureName, String delimiter) {
		String s = "";
		EStructuralFeature feature = null;

		if (elements != null && elements.size() > 0) {

			for (EStructuralFeature f : elements.get(0).eClass()
					.getEAllStructuralFeatures()) {
				if (f.getName().equals(featureName)) {
					feature = f;
					break;
				}
			}

			for (int i = 0; i < elements.size(); i++) {
				EObject e = elements.get(i);
				s += e.eGet(feature);
				if (i < elements.size() - 1)
					s += delimiter;
			}
		}

		return s;
	}

	/**
	 * Returns the name of the {@link Lifeline}'s corresponding {@link Region}
	 * <code>statemachine</code>.
	 * 
	 * @param lifeline
	 * @return the name of the corresponding state machine
	 */
	protected String getStateMachineName(Lifeline lifeline) {
		String stateMachineName = "_";
		if (lifeline.getClass_() != null
				&& lifeline.getClass_().getStatemachine() != null) {
			stateMachineName = lifeline.getClass_().getStatemachine().getName();
		}
		return stateMachineName;
	}
}
