/**
 * <copyright>
 *
 * Copyright (c) 2011 modelevolution.org
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * </copyright>
 */
package org.modelevolution.multiview.merge.engine.impl;

import java.util.Iterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.compare.diff.metamodel.DiffElement;
import org.eclipse.emf.compare.diff.metamodel.DifferenceKind;
import org.eclipse.emf.compare.match.metamodel.Side;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.EcoreUtil.Copier;
import org.modelevolution.multiview.Lifeline;
import org.modelevolution.multiview.LifelineElement;
import org.modelevolution.multiview.Message;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.ReceiveEvent;
import org.modelevolution.multiview.SendEvent;
import org.modelevolution.multiview.conflictreport.ConflictFragment;
import org.modelevolution.multiview.conflictreport.ConflictReport;
import org.modelevolution.multiview.conflictreport.ConflictReportFactory;
import org.modelevolution.multiview.conflictreport.MergeOption;
import org.modelevolution.multiview.merge.MergeAdvice;
import org.modelevolution.multiview.merge.MergePosition;
import org.modelevolution.multiview.merge.MergeType;
import org.modelevolution.multiview.merge.engine.IMergeEngine;
import org.modelversioning.conflicts.detection.impl.ThreeWayDiffProvider;
import org.modelversioning.core.util.UUIDUtil;
import org.modelversioning.merge.IMerger;
import org.modelversioning.merge.impl.MergeSpecificChangeStrategy;
import org.modelversioning.merge.impl.MergerImpl;

/**
 * @author <a href="mailto:brosch@big.tuwien.ac.at">Petra Brosch</a>
 * 
 */
public class MultiviewMergeEngine implements IMergeEngine {

	private static final Logger logger = Logger
			.getLogger("org.modelevolution.multiview");

	private ConflictReport conflictReport = null;
	private MultiviewModel mergedModel = null;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.modelevolution.multiview.merge.engine.IMergeEngine#merge(org.
	 * modelversioning.conflicts.detection.impl.ThreeWayDiffProvider,
	 * org.modelevolution.multiview.merge.MergeAdvice,
	 * org.eclipse.emf.ecore.resource.Resource)
	 */
	@Override
	public void merge(ThreeWayDiffProvider threeWayDiff,
			MergeAdvice mergeAdvice, Resource mergedModelResource) {

		// prepare merged model: copy origin model and apply all changes
		MergeSpecificChangeStrategy mergeStrategy = new MergeSpecificChangeStrategy();
		mergeStrategy.setChangesToMerge(getChangesToMerge(threeWayDiff));

		IMerger merger = new MergerImpl();
		merger.setMergeStrategy(mergeStrategy);

		merger.merge(
				org.modelversioning.conflictreport.ConflictReportFactory.eINSTANCE
						.createConflictReport(), threeWayDiff
						.getComparisonSnapshot(Side.LEFT), threeWayDiff
						.getComparisonSnapshot(Side.RIGHT),
				mergedModelResource, new NullProgressMonitor());

		mergedModel = (MultiviewModel) mergedModelResource.getContents().get(0);
	}

	@Override
	public void merge(ThreeWayDiffProvider threeWayDiff,
			MergeAdvice mergeAdvice, Resource mergedModelResource,
			Resource conflictModelResource) {

		if (conflictModelResource != null) {
			conflictReport = (ConflictReport) conflictModelResource
					.getContents().get(0);
		}

		/**
		 * Workaround for BUG in {@link ThreeWayDiffProvider} of {@link IMerger}
		 * Merge the model only for the first solution. For all succeeding
		 * solutions make a copy an resort messages.
		 */
		if (conflictReport == null
				|| conflictReport.getMergedVersions() == null
				|| conflictReport.getMergedVersions().isEmpty()) {
			merge(threeWayDiff, mergeAdvice, mergedModelResource);
		} else {
			Copier copier = new EcoreUtil.Copier();
			mergedModel = (MultiviewModel) copier.copy(conflictReport
					.getMergedVersions().get(0));
			copier.copyReferences();

			mergedModelResource.getContents().add(mergedModel);

			// copy UUIDs
			for (EObject origin : copier.keySet()) {
				UUIDUtil.copyUUID(origin, copier.get(origin));
			}
		}
		
		sortMessages(threeWayDiff, mergeAdvice);

		// prepare conflict report
		if (conflictReport != null) {
			conflictReport.getMergedVersions().add(mergedModel);

			EList<EObject> leftAddedEObjects = threeWayDiff.getAddedEObjects(
					Side.LEFT, true);
			EList<EObject> rightAddedEObjects = threeWayDiff.getAddedEObjects(
					Side.RIGHT, true);
			EList<EObject> leftUpdatedEObjects = threeWayDiff
					.getUpdatedEObjects(Side.LEFT, true);
			EList<EObject> rightUpdatedEObjects = threeWayDiff
					.getUpdatedEObjects(Side.RIGHT, true);
			EList<EObject> leftDeletedEObjects = threeWayDiff
					.getDeletedEObjects(Side.LEFT, true);
			EList<EObject> rightDeletedEObjects = threeWayDiff
					.getDeletedEObjects(Side.RIGHT, true);

			for (ConflictFragment c : conflictReport.getConflicts()) {
				Message lastOriginMsgCopy = null;
				Message nextOriginMessageCopy = null;

				MergeOption mergeOption = ConflictReportFactory.eINSTANCE
						.createMergeOption();

				if (c.getLastOrigin() != null) {
					lastOriginMsgCopy = findMessageByFragment(c.getLastOrigin());
				}

				if (c.getNextOrigin() != null) {
					nextOriginMessageCopy = findMessageByFragment(c
							.getNextOrigin());
				}

				Message startMsg = (lastOriginMsgCopy == null) ? mergedModel
						.getSequenceview().getOrderedMessages().get(0)
						: mergedModel
								.getSequenceview()
								.getOrderedMessages()
								.get(mergedModel.getSequenceview()
										.getOrderedMessages()
										.indexOf(lastOriginMsgCopy) + 1);

				Message endMsg = (nextOriginMessageCopy == null) ? mergedModel
						.getSequenceview()
						.getOrderedMessages()
						.get(mergedModel.getSequenceview().getMessages().size() - 1)
						: mergedModel
								.getSequenceview()
								.getOrderedMessages()
								.get(mergedModel.getSequenceview()
										.getOrderedMessages()
										.indexOf(nextOriginMessageCopy) - 1);

				mergeOption.setStartMessage(startMsg);
				mergeOption.setEndMessage(endMsg);

				EList<Message> orderedMessages = mergedModel.getSequenceview()
						.getOrderedMessages();
				for (int i = orderedMessages.indexOf(startMsg); i <= orderedMessages
						.indexOf(endMsg); i++) {
					Message m = orderedMessages.get(i);

					if (findEObjectByFragment(m, leftAddedEObjects) != null) {
						mergeOption.getLeftAddedMessages().add(m);
					} else if (findEObjectByFragment(m, rightAddedEObjects) != null) {
						mergeOption.getRightAddedMessages().add(m);
					} else if (findEObjectByFragment(m, leftDeletedEObjects) != null) {
						// FIXME we should search for origin counterparts within
						// deletedObjects
						mergeOption.getLeftDeletedMessages().add(m);
					} else if (findEObjectByFragment(m, rightDeletedEObjects) != null) {
						// FIXME we should search for origin counterparts within
						// deletedObjects
						mergeOption.getRightDeletedMessages().add(m);
					} else if (findEObjectByFragment(m, leftUpdatedEObjects) != null) {
						mergeOption.getLeftUpdatedMessages().add(m);
					} else if (findEObjectByFragment(m, rightUpdatedEObjects) != null) {
						mergeOption.getRightUpdatedMessages().add(m);
					}
				}

				c.getMergeOptions().add(mergeOption);
			}
		}
	}

	/**
	 * Sorts all {@link Message}s of the merged model according to the given
	 * {@link MergeAdvice}.
	 * 
	 * @param threeWayDiff
	 *            The diffProvider to use for looking up messages.
	 * @param mergeAdvice
	 *            The mergeAdvice holding the sort order in question.
	 */
	private void sortMessages(ThreeWayDiffProvider threeWayDiff,
			MergeAdvice mergeAdvice) {
		// sort messages
		for (MergePosition mp : mergeAdvice.getMergePositions()) {
			Message msg = null;
			Message msgCopy = null;

			if (mp.getType().equals(MergeType.O)) {
				msg = ((MultiviewModel) threeWayDiff.getOriginModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.L)) {
				msg = ((MultiviewModel) threeWayDiff.getLeftModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.R)) {
				msg = ((MultiviewModel) threeWayDiff.getRightModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else {
				logger.log(Level.SEVERE,
						"Message with mergeType {0} on index {1} not found.",
						new Object[] { mp.getType(), mp.getIndex() });
			}

			msgCopy = findMessageByFragment(msg);

			SendEvent sendEventCopy = msgCopy.getSender();
			ReceiveEvent receiveEventCopy = msgCopy.getReceiver();

			Lifeline senderLifelineCopy = sendEventCopy.getLifeline();
			Lifeline receiverLifelineCopy = receiveEventCopy.getLifeline();

			senderLifelineCopy.getElements().move(
					senderLifelineCopy.getElements().size() - 1, sendEventCopy);
			receiverLifelineCopy.getElements().move(
					receiverLifelineCopy.getElements().size() - 1,
					receiveEventCopy);
		}
	}
	
	/**
	 * @param threeWayDiff
	 * @param mergeAdvice
	 * @param merger
	 */
	private void sortMessages(ThreeWayDiffProvider threeWayDiff,
			MergeAdvice mergeAdvice, IMerger merger) {
		// sort messages
		for (MergePosition mp : mergeAdvice.getMergePositions()) {
			Message msg = null;
			Message msgCopy = null;

			if (mp.getType().equals(MergeType.O)) {
				msg = ((MultiviewModel) threeWayDiff.getOriginModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.L)) {
				msg = ((MultiviewModel) threeWayDiff.getLeftModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.R)) {
				msg = ((MultiviewModel) threeWayDiff.getRightModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else {
				logger.log(Level.SEVERE,
						"Message with mergeType {0} on index {1} not found.",
						new Object[] { mp.getType(), mp.getIndex() });
			}

			msgCopy = (Message) merger.getCorrespondingMergedObject(msg);

			if (msgCopy == null) {
				msgCopy = findMessageByFragment(msg);
			}

			SendEvent sendEventCopy = msgCopy.getSender();
			ReceiveEvent receiveEventCopy = msgCopy.getReceiver();

			Lifeline senderLifelineCopy = sendEventCopy.getLifeline();
			Lifeline receiverLifelineCopy = receiveEventCopy.getLifeline();

			senderLifelineCopy.getElements().move(
					senderLifelineCopy.getElements().size() - 1, sendEventCopy);
			receiverLifelineCopy.getElements().move(
					receiverLifelineCopy.getElements().size() - 1,
					receiveEventCopy);
		}

		// check sorting of messages
		for (MergePosition mp : mergeAdvice.getMergePositions()) {
			Message msg = null;
			Message msgCopy = null;

			if (mp.getType().equals(MergeType.O)) {
				msg = ((MultiviewModel) threeWayDiff.getOriginModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.L)) {
				msg = ((MultiviewModel) threeWayDiff.getLeftModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.R)) {
				msg = ((MultiviewModel) threeWayDiff.getRightModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else {
				logger.log(Level.SEVERE,
						"Message with mergeType {0} on index {1} not found.",
						new Object[] { mp.getType(), mp.getIndex() });
			}

			msgCopy = (Message) merger.getCorrespondingMergedObject(msg);

			if (msgCopy == null) {
				msgCopy = findMessageByFragment(msg);
			}

			if (!mergedModel.getSequenceview().getOrderedMessages()
					.get(mp.getIndex() - 1).equals(msgCopy)) {
				logger.log(
						Level.SEVERE,
						"Message {0} found on Position {2}, but Message {1} expected.",
						new Object[] {
								mergedModel.getSequenceview()
										.getOrderedMessages()
										.get(mp.getIndex() - 1), msgCopy,
								mp.getIndex() - 1 });
			}
		}
	}

	/**
	 * Checks whether the order of the {@link Message}s in the merged model
	 * adhere to the given {@link MergeAdvice}.
	 * 
	 * @param threeWayDiff
	 *            The diffProvider to use for looking up messages.
	 * @param mergeAdvice
	 *            The mergeAdvice holding the sort order in question.
	 */
	private boolean checkSorting(ThreeWayDiffProvider threeWayDiff,
			MergeAdvice mergeAdvice) {
		// check sorting of messages
		for (MergePosition mp : mergeAdvice.getMergePositions()) {
			Message msg = null;
			Message msgCopy = null;
			Message msgMerged = null;

			if (mp.getType().equals(MergeType.O)) {
				msg = ((MultiviewModel) threeWayDiff.getOriginModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.L)) {
				msg = ((MultiviewModel) threeWayDiff.getLeftModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else if (mp.getType().equals(MergeType.R)) {
				msg = ((MultiviewModel) threeWayDiff.getRightModel().get(0))
						.getSequenceview().getOrderedMessages()
						.get(mp.getIndex() - 1);
			} else {
				logger.log(Level.SEVERE,
						"Message with mergeType {0} on index {1} not found.",
						new Object[] { mp.getType(), mp.getIndex() });
				return false;
			}

			msgCopy = findMessageByFragment(msg);
			msgMerged = mergedModel.getSequenceview().getOrderedMessages()
					.get(mp.getIndex() - 1);

			if (!msgMerged.equals(msgCopy)) {
				logger.log(
						Level.SEVERE,
						"Message {0} found on Position {2}, but Message {1} expected.",
						new Object[] {
								mergedModel.getSequenceview()
										.getOrderedMessages()
										.get(mp.getIndex() - 1), msgCopy,
								mp.getIndex() - 1 });
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns all {@link DiffElement}s except for {@link LifelineElement}s and
	 * {@link Message}s.
	 * 
	 * @param threeWayDiff
	 *            The {@link ThreeWayDiffProvider} to use.
	 * @return the {link DiffElement}s
	 */
	private EList<DiffElement> getChangesToMerge(
			ThreeWayDiffProvider threeWayDiff) {
		EList<DiffElement> diffs = new BasicEList<DiffElement>();
		
		EList<EObject> allLeftImplicitlyAddedObjects = threeWayDiff.getAddedEObjects(Side.LEFT, true);
		EList<EObject> allRightImplicitlyAddedObjects = threeWayDiff.getAddedEObjects(Side.RIGHT, true);

		for (DiffElement diff : threeWayDiff
				.getEffectiveDiffElements(Side.LEFT)) {
			diffs.add(diff);
			
//			// search for implicitly added Elements
//			if (diff.getKind().equals(DifferenceKind.ADDITION)) {
//				EObject addedObject = DiffUtil.getLeftElement(diff);
//				if (allLeftImplicitlyAddedObjects.contains(addedObject)) {
//					for (EObject)
//				}
//			}
		}
		
		for (DiffElement diff : threeWayDiff
				.getEffectiveDiffElements(Side.RIGHT)) {
			diffs.add(diff);
			
			// search for implicitly added Elements
		}
		
		return diffs;
	}

	/**
	 * Finds the corresponding {@link Message} in the merged model based on its
	 * fragment.
	 * 
	 * @param msg
	 *            the {@link Message} to search
	 * @return the matching {@link Message} in the merged model
	 */
	private Message findMessageByFragment(Message msg) {
		Iterator<Message> i = mergedModel.getSequenceview().getMessages()
				.iterator();
		while (i.hasNext()) {
			Message o = i.next();
			if (EcoreUtil.getURI(o).fragment()
					.equals(EcoreUtil.getURI(msg).fragment())
					&& o instanceof Message) {
				return o;
			}
		}

		return null;
	}

/**
	 * Finds the corresponding {@link EObject} in the merged model based on its fragment.   
	 * @param eObject
	 * 			the {@link EObject to search
	 * @param list
	 * 			the list of parent elements to search in
	 * @return the matching {@link EObject} in the merged model
	 */
	private EObject findEObjectByFragment(EObject eObject, EList<EObject> list) {
		for (EObject o : list) {
			if (EcoreUtil.getURI(o).fragment()
					.equals(EcoreUtil.getURI(eObject).fragment())) {
				return o;
			}
		}

		return null;
	}
}