package org.modelevolution.multiview.modelgenerator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Random;
import java.util.Set;


import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.modelevolution.multiview.ClassView;
import org.modelevolution.multiview.Lifeline;
import org.modelevolution.multiview.LifelineElement;
import org.modelevolution.multiview.Message;
import org.modelevolution.multiview.MultiviewFactory;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.MultiviewPackage;
import org.modelevolution.multiview.ReceiveEvent;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.SendEvent;
import org.modelevolution.multiview.SequenceView;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.StateView;
import org.modelevolution.multiview.Symbol;
import org.modelevolution.multiview.Transition;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;

public class SequenceGenerator {
	
	public static final String FILE_EXTENSION = "mvml";
	public static final boolean IGNORE_EFFECTS = false;
	private static final boolean DEBUG = false;



	private Random random;
	private Resource resource;
	private ResourceSet resourceSet;
	private ClassView classView;
	private SequenceView sequenceView;
	private Map<Lifeline,State> currentStates;
	private Set<Symbol> effects;
	private File file;
	private MultiviewModel mvmodel;
	
/**
 * Generates sequence diagram from set of statemachines and creates new .mvml file
 * @param file
 */
	public SequenceGenerator(File file, Random random){
		this.random = random;
		this.file = file;
		
		EPackage.Registry.INSTANCE.put(
				MultiviewPackage.eINSTANCE.getNsURI(),
				MultiviewPackage.eINSTANCE);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
				FILE_EXTENSION, new UUIDResourceFactoryImpl());

		
		resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put(FILE_EXTENSION, new UUIDResourceFactoryImpl());

		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
				.put("xmi", new UUIDResourceFactoryImpl());

		URI fileURI_id = URI.createFileURI(file.getAbsolutePath());
		resource = resourceSet.getResource(fileURI_id, true);
		mvmodel = (MultiviewModel) resource.getContents().get(0);
		

		if(mvmodel.getStateview() == null || mvmodel.getClassview() == null){
			System.err.println("Cannot create SequenceGenerator because there is no State Machine");
			return;
		}
		
		classView = mvmodel.getClassview();
		
	// create new sequence diagram
		if(mvmodel.getSequenceview() == null){
			sequenceView = MultiviewFactory.eINSTANCE.createSequenceView();
			mvmodel.setSequenceview(sequenceView);
		}
	// create new version of sequence diagram
		else{
			sequenceView = mvmodel.getSequenceview();
			
			// make sure each lifeline has a statemachine
			for(Lifeline lifeline : sequenceView.getLifelines()){
				if(lifeline.getClass_() == null || lifeline.getClass_().getStatemachine() == null){
					lifeline.initDummyStatemachine();
				}
			}
		}
		
		effects = new HashSet<Symbol>();
		
		StateView stateView = mvmodel.getStateview();
		for(Region statemachine : stateView.getStatemachines()){
			for(State state : statemachine.getStates()){
				for(Transition transition : state.getIncoming()){
					for(Symbol effect : transition.getEffects()){
						effects.add(effect);
					}
				}
			}
		}
			
	}
	
	public File generateSequence(int nrLifelines, int nrMessages, boolean copy, String filename){
		
		System.out.println("Generating "+nrLifelines+" lifelines");
		generateLifelines(nrLifelines);
		
		setCurrentStates();
		
		System.out.println("Generating "+nrMessages+" messages");
		generateMessages(nrMessages);

		if(copy){
			File newFile = new File(filename);
			URI fileURI = URI.createFileURI(newFile.getAbsolutePath());
			Resource newResource = resourceSet.createResource(fileURI);
			newResource.getContents().add(mvmodel);
			try {
				newResource.save(Collections.EMPTY_MAP);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			
			return newFile;
			
		}
		else{
			try {
				resource.save(Collections.EMPTY_MAP);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		
		}	
		return file;
	}
	
	
	private void generateMessages(int nrMessages) {
				int i=0;
				Message message = getMessage();
				if(message != null) sequenceView.getMessages().add(message);

				while(i<nrMessages && message != null){	
					message = getMessage();
					if(message != null) sequenceView.getMessages().add(message);
					i++;
				}
				
				if(message == null && i<nrMessages) System.err.println(" Due to deadlock only "+i+" messages out of "+nrMessages+" could be generated.");		
	}
	

	private Message getMessage() {
		
		//make sure no message is attached to two empty lifelines
		boolean emptyOK = true; 
				
		if(sequenceView.getMessages().size() > 0)
								emptyOK = random.nextBoolean();
		
	// check if message is possible
		Lifeline receiveLL = getReceiverLifeline(emptyOK);
		if(receiveLL == null) return null;

		Symbol body = getSymbol(receiveLL);
		if(body == null) return null;
		
		if(receiveLL.getElements().size() == 1 && sequenceView.getMessages().size() > 1) emptyOK = false;
		else emptyOK = true;
		
		Lifeline sendLL = getSenderLifeline(body,emptyOK);
		if(sendLL == null) return null;
		
	// create and register events and message
		SendEvent sendEvent = MultiviewFactory.eINSTANCE.createSendEvent();
		sendEvent.setName(getSendEventName(body, sendLL));
		sendEvent.setLifeline(sendLL);

		ReceiveEvent receiveEvent = MultiviewFactory.eINSTANCE.createReceiveEvent();
		receiveEvent.setName(getReceiveEventName(body, receiveLL));
		receiveEvent.setLifeline(receiveLL);
		
		Message message = MultiviewFactory.eINSTANCE.createMessage();
		message.setReceiver(receiveEvent);
		message.setBody(body);
		message.setSender(sendEvent);

		return message;

	}
	
	private Lifeline getReceiverLifeline(boolean emptyOK) {

		List<Lifeline> possibleLifelines = new ArrayList<Lifeline>();

		// get possible lifelines (not deadlocked)
		for (Lifeline lifeline : currentStates.keySet()){
			if (currentStates.get(lifeline) == null || currentStates.get(lifeline).getOutgoing().size() > 0){
				if(emptyOK || lifeline.getElements().size() > 0 ){
					possibleLifelines.add(lifeline);
				}
			}
		}
		
		if(possibleLifelines.size() == 0) return null;
		
		// randomly choose lifeline
		Lifeline lifeline = possibleLifelines.get(random.nextInt(possibleLifelines.size()));
		if(DEBUG) System.out.println("\return "+lifeline.getName());

		return lifeline;
	}
	
	private Lifeline getSenderLifeline(Symbol symbol, boolean emptyOK) {

		List<Lifeline> possibleLifelines = new ArrayList<Lifeline>();
		
		// get possible lifelines (effect exists in statemachine)
		for (Lifeline lifeline : sequenceView.getLifelines()){
			if (IGNORE_EFFECTS || effectInStatemachine(symbol,lifeline.getClass_().getStatemachine())){
				if(emptyOK || lifeline.getElements().size() > 0 ){
					possibleLifelines.add(lifeline);
				}
			}
		}
		
		if(possibleLifelines.size() == 0) return null;
		
		// randomly choose lifeline
		Lifeline lifeline = possibleLifelines.get(random.nextInt(possibleLifelines.size()));
		if(DEBUG) System.out.println("\return "+lifeline.getName());

		return lifeline;
	}
	
	private boolean effectInStatemachine(Symbol effect,Region statemachine) {
		for (State state : statemachine.getStates()){
			for (Transition transition : state.getIncoming()){
				if(transition.getEffects().contains(effect)) return true;
			}
		}

		return false;
	}

	/**
	 * initialize hashmap to track states of each lifeline's statemachine according to message sequences
	 */
	private void setCurrentStates() {
				currentStates = new HashMap<Lifeline,State>();
				for (Lifeline lifeline : sequenceView.getLifelines()){
					
					if(DEBUG) System.out.print("Put "+lifeline.getName()+"\t");			
					if(lifeline.getElements().isEmpty()) 	{
						currentStates.put(lifeline, null);
						if(DEBUG) System.out.println("null");			

					}
					else{
						State state = getCurrentState(lifeline);
						if(DEBUG) 
							if(state == null) System.out.println("false null");		
							else System.out.println(state.getName());			
						currentStates.put(lifeline, state);
					}
				}			
	}

	private void generateLifelines(int nrLifelines) {
		// TODO statemachine.getStates() should be shuffled for more randomness
				Iterator<org.modelevolution.multiview.Class> iterator = classView.getClasses().iterator();

				// make sure each statemachine is instantiated at least once (unless nrLifelines is smaller than the number of statemachines)
				int i=0;
				while(i<nrLifelines && iterator.hasNext()){
					
					org.modelevolution.multiview.Class class_ = iterator.next();
					
					if(class_.getStatemachine() != null){
						Lifeline lifeline = MultiviewFactory.eINSTANCE.createLifeline();
						lifeline.setClass(class_);
						lifeline.setName(getLLName(class_.getName()));
						sequenceView.getLifelines().add(lifeline);
						
//						System.out.println("Added "+lifeline.getName());

					}
					i++;
				}
				
				// for the remaining lifelines choose statemachine randomly
				while(i<nrLifelines){
					
					org.modelevolution.multiview.Class class_ = classView.getClasses().get(random.nextInt(classView.getClasses().size()));
					
					if(class_.getStatemachine() != null){
						Lifeline lifeline = MultiviewFactory.eINSTANCE.createLifeline();
						lifeline.setClass(class_);
						lifeline.setName(getLLName(class_.getName()));
						sequenceView.getLifelines().add(lifeline);
						
//						System.out.println("Added "+lifeline.getName());

					}
					i++;
				
				}		
	}
	
	private String getLLName(String name) {
		
		//return name+"-"+Long.toString(System.currentTimeMillis());
		return name+"-"+sequenceView.getLifelines().size();
	}
	
	private String getReceiveEventName(Symbol body, Lifeline lifeline) {
		return "RCV-"+body.getName()+"-"+lifeline.getElements().size();

	}

	private String getSendEventName(Symbol body, Lifeline lifeline) {
		
		return "SND-"+body.getName()+"-"+lifeline.getElements().size();
	}

	private State getCurrentState(Lifeline lifeline) {
		
		List<Message> messages = getMessages(lifeline);
		
		Region sm = lifeline.getClass_().getStatemachine();

		if(messages.isEmpty()){
			return null;
		}
		
		List<State> startStates = getStartStates(sm, messages.get(0));

		if(startStates.isEmpty()){
			return null;
		}
		
		State state = searchPaths(startStates, messages);
		
		if(DEBUG) if(state == null)System.out.println("getCurrentState state null");
		
		return state;
		
	}
	
	/*
	 * FIXME Maybe this should be somewhere else
	 * FIXME this method also in ModelVerifier.java
	 * FIXME REFACTOR!
	 */
	private List<Message> getMessages(Lifeline lifeline) {

		List<Message> messages = new ArrayList<Message>();

		if (lifeline != null) {
			for (LifelineElement le : lifeline.getElements()) {
				if (le instanceof ReceiveEvent) {
					messages.add(((ReceiveEvent)le).getMessage());
				}
			}

		}

		return messages;
	}
	
	/*
	 * FIXME this method also in ModelVerifier.java
	 * FIXME REFACTOR!
	 */
	private List<State> getStartStates(Region sm, Message m) {

		List<State> states = new ArrayList<State>();

		for (State state : sm.getStates()) {

			for (Transition t : state.getIncoming()) {

				if (t.getSource() != null) {

					if (t.getTrigger().equals(m.getBody())) {
						states.add(state);
					}
				}
			}

		}
		return states;

	}
	
	/*
	 * FIXME this method also in ModelVerifier.java
	 * FIXME REFACTOR!
	 */
	private State searchPaths(List<State> startStates,
			List<Message> messages) {

		State state = null;
		Collections.shuffle(startStates);
		Iterator<State> iterator = startStates.iterator();
		
		while(state==null && iterator.hasNext()){
			state = searchPath(iterator.next(), messages);
		}

		return state;
	}

	/*
	 * FIXME this method also in ModelVerifier.java
	 * FIXME REFACTOR!
	 */
	private State searchPath(State start, List<Message> messages) {

		int index = 1;
		State currState = start;

		while (index < messages.size()) {

			Message currMessage = messages.get(index);

			currState = getNextState(currState, currMessage);

			if (currState == null) {
				return currState;
			} else {
				index++;
			}

		}

		return currState;
	}
	
	/*
	 * FIXME this method also in ModelVerifier.java
	 * FIXME REFACTOR!
	 */
	private State getNextState(State state, Message m) {

		State next = null;

		for (Transition outTrans : state.getOutgoing()) {

			if (outTrans.getTrigger().equals(m.getBody())) {
				next = outTrans.getTarget();
			}
		}

		return next;
	}


	private Symbol getSymbol(Lifeline receiveLL) {

		State state = currentStates.get(receiveLL);
		List<Transition> possibleTransitions = new ArrayList<Transition>();
		
		List<Transition> transitions;
		if(state == null){
			transitions = getTransitions(receiveLL.getClass_().getStatemachine());
		}
		else{
			transitions = state.getOutgoing();
		}
		
		if(!IGNORE_EFFECTS )
			for(Transition transition : transitions){
				if(effects.contains(transition.getTrigger()))
					possibleTransitions.add(transition);
			}
		else possibleTransitions = transitions;
		
		if(possibleTransitions.size() == 0) return null;
		
		Transition transition = possibleTransitions.get(random.nextInt(possibleTransitions.size()));

		// update current state for lifeline
		currentStates.put(receiveLL, transition.getTarget());
		//System.out.println("UPDATE currentStates "+receiveLL.getName()+" val "+transition.getTarget().getName());
		
		return transition.getTrigger();
		
	}

	private List<Transition> getTransitions(Region statemachine){
		List<Transition> transitions = new ArrayList<Transition>();
		
		for(State state : statemachine.getStates()){
			for(Transition transition : state.getIncoming()){
				transitions.add(transition);
			}
		}
		
		return transitions;
	}

	/**
	 * @param args
	 */
//	public static void main(String[] args) {
//		int nrLifelines = Integer.parseInt(args[0]);
//		int nrMessages = Integer.parseInt(args[1]);
//
//		File file = null;
//				
//		if (args.length > 1) {
//			file = new File(args[2]);			
//		}
//		
//		SequenceGenerator sg = new SequenceGenerator(file, new Random(System.currentTimeMillis()));
//		sg.generateSequence(nrLifelines, nrMessages);
//		
//		System.out.println("DONE!");
//
//	}
}
