package org.modelevolution.multiview.modelgenerator;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Random;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.modelevolution.multiview.ClassView;
import org.modelevolution.multiview.MultiviewFactory;
import org.modelevolution.multiview.MultiviewModel;
import org.modelevolution.multiview.MultiviewPackage;
import org.modelevolution.multiview.Region;
import org.modelevolution.multiview.State;
import org.modelevolution.multiview.StateView;
import org.modelevolution.multiview.Symbol;
import org.modelevolution.multiview.Transition;
import org.modelversioning.core.impl.UUIDResourceFactoryImpl;
import org.modelevolution.multiview.modelgenerator.Bounds;


public class StatemachineGenerator {

	
	private static final String FILE_EXTENSION = "mvml";
	private static final String PATH = "~/Documents/SVN-FAME/dev/test/org.modelevolution.core.diff.encoding.test/models/";
	private static final double PROB_NEW_SYMBOL = 0.00000;
	private static final int MAX_NUMBER_EFFECTS = 4;


	private Random random;
	private EList<Symbol> alphabet;
	private MultiviewModel mvmodel;
	private StateView stateView;	
	private ClassView classView;
	private Resource resource;
	private ResourceSet resourceSet;
	private File file;
	
	public StatemachineGenerator(String filename, Random random){
		this.random = random;
		
		file = new File(filename);
		URI fileURI_id = URI.createFileURI(file.getAbsolutePath());
		resourceSet = getResourceSet();
		resource = resourceSet.createResource(fileURI_id);
		mvmodel = MultiviewFactory.eINSTANCE.createMultiviewModel();
		resource.getContents().add(mvmodel);
		setUp();
	}

	
	public StatemachineGenerator(File file, Random random){
		this.random = random;
		this.file = file;
		
		URI fileURI_id = URI.createFileURI(file.getAbsolutePath());
		resourceSet = getResourceSet();
		resource = resourceSet.getResource(fileURI_id, true);
		mvmodel = (MultiviewModel) resource.getContents().get(0);		
		setUp();
	}
	
	private void setUp() {
		// create or retrieve class view

		if(mvmodel.getClassview() == null){
			classView = MultiviewFactory.eINSTANCE.createClassView();
			mvmodel.setClassview(classView);
		}
		else classView = mvmodel.getClassview();
		
		// create or retrieve state view and update alphabet
		if(mvmodel.getStateview() == null){
			stateView = MultiviewFactory.eINSTANCE.createStateView();
			mvmodel.setStateview(stateView);
		}
		else{
			stateView = mvmodel.getStateview();
		}
		instantiateAlphabet(stateView);
		alphabet = mvmodel.getStateview().getAlphabet();	
	}


	private ResourceSet getResourceSet(){
		
		if(resourceSet == null){
			EPackage.Registry.INSTANCE.put(
					MultiviewPackage.eINSTANCE.getNsURI(),
					MultiviewPackage.eINSTANCE);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(
					FILE_EXTENSION, new UUIDResourceFactoryImpl());

			
			resourceSet = new ResourceSetImpl();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
					.put(FILE_EXTENSION, new UUIDResourceFactoryImpl());

			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
					.put("xmi", new UUIDResourceFactoryImpl());
		}

		return resourceSet;
	}
	
	private void instantiateAlphabet(StateView stateView) {
		if(stateView.getAlphabet().isEmpty()){
			Symbol symbol1 = MultiviewFactory.eINSTANCE.createSymbol();
			symbol1.setName("1");
			stateView.getAlphabet().add(symbol1);
			Symbol symbol2 = MultiviewFactory.eINSTANCE.createSymbol();
			symbol2.setName("2");
			stateView.getAlphabet().add(symbol2);
		}
		
	}
	
	public File generateStatemachine(Bounds states, Bounds transitions, boolean copy, String filename){
		
		int nrStates = random.nextInt(states.getUpper() - states.getLower() + 1) + states.getLower();
		int nrTransitions = random.nextInt(transitions.getUpper() - transitions.getLower() + 1) + transitions.getLower();
		return generateStatemachine(nrStates, nrTransitions, copy, filename);
	}

	public File generateStatemachine(int nrStates, int nrTransitions, boolean copy, String filename){
		String name = Long.toString(System.currentTimeMillis());

		System.out.println("Generating "+nrStates+" states");
		System.out.println("Generating "+nrTransitions+" transitions");

		//get statemachine
		Region statemachine = getRandomStatemachine(nrStates, nrTransitions,stateView);
		statemachine.setName(name);
		
		// create and add class
		org.modelevolution.multiview.Class class_ = MultiviewFactory.eINSTANCE.createClass();
		class_.setName(name);
		classView.getClasses().add(class_);
			
		
		// add statemachine
		class_.setStatemachine(statemachine);
		stateView.getStatemachines().add(statemachine);
		
		if(copy){
			File newFile = new File(filename);
			URI fileURI = URI.createFileURI(newFile.getAbsolutePath());
			Resource newResource = resourceSet.createResource(fileURI);
			newResource.getContents().add(mvmodel);
			try {
				newResource.save(Collections.EMPTY_MAP);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			
			return newFile;
			
		}
		else{
			try {
				resource.save(Collections.EMPTY_MAP);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		
		}

		return file;
		
	}
	
	private Region getRandomStatemachine(int nrStates, int nrTransitions,
			StateView stateView) {
		Region statemachine = MultiviewFactory.eINSTANCE.createRegion();
		
		System.out.println("Generating "+nrStates+" states");

		// generate states
		for(int i=0; i<nrStates; i++){
			State state = MultiviewFactory.eINSTANCE.createState();
			state.setName(Integer.toString(i));
			statemachine.getStates().add(state);
		}
		
		System.out.println("Generating "+nrTransitions+" transitions");

		// generate transitions
		// TODO statemachine.getStates() should be shuffled for more randomness
		Iterator<State> iterator = statemachine.getStates().iterator();
		
		// make sure each state is attached to at least one transition (if there are less states than transitions)
		int i=0;
		int nrEffects = 0;
		while(i<nrTransitions && iterator.hasNext()){
			
			Transition transition = MultiviewFactory.eINSTANCE.createTransition();
			if(MAX_NUMBER_EFFECTS > 0) nrEffects = random.nextInt(MAX_NUMBER_EFFECTS);
			
		// randomly choose whether current state is source or target, and choose the other randomly
			if(random.nextDouble() < 0.5){
				transition.setSource(iterator.next());
				transition.setTarget(statemachine.getStates().get(random.nextInt(statemachine.getStates().size())));
			}
			else{
				transition.setTarget(iterator.next());
				transition.setSource(statemachine.getStates().get(random.nextInt(statemachine.getStates().size())));
			}
			
			transition.setTrigger(getSymbol(transition.getSource(),false));
			
			for(int j=0; j<nrEffects; j++ ){
				transition.getEffects().add(getSymbol(transition.getSource(),false));
			}
			i++;
		}
		
		while(i<nrTransitions){
			
			Transition transition = MultiviewFactory.eINSTANCE.createTransition();
			if(MAX_NUMBER_EFFECTS > 0) nrEffects = random.nextInt(MAX_NUMBER_EFFECTS);
			
		// randomly choose both source and target  

				transition.setTarget(statemachine.getStates().get(random.nextInt(statemachine.getStates().size())));
				transition.setSource(statemachine.getStates().get(random.nextInt(statemachine.getStates().size())));
			
			transition.setTrigger(getSymbol(transition.getSource(),false));
			
			for(int j=0; j<nrEffects; j++ ){
				transition.getEffects().add(getSymbol(transition.getSource(),false));
			}
			i++;
		}
		
		return statemachine;
	}
	
	private Symbol getSymbol(State origin, boolean createNew){

		Symbol symbol;
		
		// randomly decide whether to use an existing symbol or choose it randomly from alphabet
		if(createNew || random.nextDouble() < PROB_NEW_SYMBOL){
			
			// new symbol												
			int label = alphabet.size() +1;
			symbol = MultiviewFactory.eINSTANCE.createSymbol();
			symbol.setName(Integer.toString(label));
			alphabet.add(symbol);
		}
		else{
			// random symbol from alphabet
			symbol = alphabet.get(random.nextInt(alphabet.size()));
		}
		
		// make sure it's deterministic
		for (Transition trans : origin.getOutgoing()){
			if(trans.getTrigger() != null)
				if (trans.getTrigger().equals(symbol)) {
					return getSymbol(origin, true);
				}
		}

		return symbol;
	}
	

	/**
	 * @param args
	 * args[0] number of states
	 * args[1] number of transitions
	 * args[2] file
	 */
//	public static void main(String[] args) {
//		
//		int nrStates = Integer.parseInt(args[0]);
//		int nrTransitions = Integer.parseInt(args[1]);
//		File file = null;
//				
//		if (args.length > 2) {
//			file = new File(args[2]);			
//		}
//		
//		StatemachineGenerator sg = new StatemachineGenerator(file, new Random());
//		sg.generateStatemachine(nrStates, nrTransitions);
//		
//		System.out.println("DONE!");
//
//
//	}

}
