


package at.jku.fmv.qbf.nnf.formula;



import java.util.ArrayList;
import java.util.List;

/**
 * Data Structure of a QBF in Negation Normal Form
 * Note: 	Connectives are n-ary (incl. 0)
 * 			a variable is only quantfied once (i.e., formula is cleansed)
 *
 */
public class Formula {
	
	private Formula superFormula = null;		// direct superformula in formula tree
	private CType connective;					// type of formula: conjunction, disjunction
	private Quantifiers quantifiers;			// direct quantifier prefix for this formula
	private boolean isPushed = false;
	
	private ArrayList<Formula> subFormulas;		// complex subformulas (i.e., no single literals)
	
	private ArrayList<VOccurrence> plits;		// list of positive literals
	private ArrayList<VOccurrence> nlits;		// list of negative literals
	private TseitinVar tseitinVar;
	
	
	
	/**
	 * initializes the new formula (contains 0 literals, 0 complex subformulas, 
	 * 								empty quantifier prefix, connective type not set)
	 */
	public Formula() {
		subFormulas = new ArrayList<Formula>();
		quantifiers = new Quantifiers(this);
		plits = new ArrayList<VOccurrence>();
		nlits = new ArrayList<VOccurrence>();
		tseitinVar = null;

		
	}
	
	public boolean comp(Formula f) {
		
		//TODO commutativity
		
		if (f.getQuantifiers().size() != 0) return false;
		if (getQuantifiers().size() != 0) return false;
		if (f.getConnective() != getConnective()) return false;
		
		if (f.getPosLits().size() != plits.size()) return false;
		if (f.getNegLits().size() != nlits.size()) return false;
		if (f.getSubformulas().size() != getSubformulas().size()) return false;

		for (int i = 0; i < f.getPosLits().size(); i++) {
			if (f.getPosLits().get(i).getVariable() != plits.get(i).getVariable()) return false;
		}
		
		for (int i = 0; i < f.getNegLits().size(); i++) {
			if (f.getNegLits().get(i).getVariable() != nlits.get(i).getVariable()) return false;
		}
		
		for (int i = 0; i < f.getSubformulas().size(); i++) {
			if (f.getSubformulas().get(i).getTseitinVar().getName() != subFormulas.get(i).getTseitinVar().getName()) return false;
		}
		
		return true;

	}
	
	public void unpush() {
		isPushed = false;
	}
	
	public void push () {
		isPushed = true;
	}
	
	public boolean isPushed () {
		return isPushed;
	}
	
	public void setTseitinVar(TseitinVar t) {
		this.tseitinVar = t;
	}
	
	public TseitinVar getTseitinVar() {
		return tseitinVar;
	}
	
	
	/**
	 * sets the type of the connective of this formula
	 * @param connective type of the connective (disjunction, conjunction)
	 */
	public void setConnective(CType connective) {
		this.connective = connective;
	}
	

	/**
	 * returns all direct subformulas
	 * @return list of all direct subformulas
	 */
	public List<Formula> getSubformulas() {
		return subFormulas;
	}
	
	
	/**
	 * returns all positive literals directly contained in this formula
	 * @return list of positive literals
	 */
	public List<VOccurrence>getPosLits() {
		return plits;
	}
	
	
	/**
	 * returns all negative literals directly contained in this formula
	 * @return list of negative literals
	 */
	public List<VOccurrence>getNegLits() {
		return nlits;
	}
	

	/**
	 * returns the direct quantifier prefix of this formula
	 * @return the quantifiers of this formula
	 */
	public Quantifiers getQuantifiers() {
		return quantifiers;
	}
	
	
	/**
	 * adds a positive occurrence of a literal to this formula
	 * @param v positive literal occurrence
	 */
	public void addPlit(VOccurrence v) {
		plits.add(v);
		v.setFormulaPosition(this,true);
	}
	
	
	/**
	 * adds a negative occurrence of a literal to this formula
	 * @param v negative literal occurrence
	 */
	public void addNlit(VOccurrence v) {
		nlits.add(v);
		v.setFormulaPosition(this,false);
	}
	
	
	/**
	 * adds a subformula to this formula
	 * @param subFormula
	 */
	public void addSubFormula(Formula subFormula){
		
		//assert (quantifiers.size() != 0) || (getConnective() != subFormula.getConnective()) : "subformula has same connective as superformula";
		
		subFormula.setSuperFormula(this);
		subFormulas.add(subFormula);
	}
	
	
	/**
	 * set the superformula of this formula
	 * @param sf the superformula to be set
	 */
	public void setSuperFormula(Formula sf) {
		this.superFormula = sf;
	}
	
	
	/**
	 * returns the superformula of this formula
	 * @return the superformula of this formula
	 */
	public Formula getSuperFormula() {
		return superFormula;
	}
	
	
	/**
	 * returns the connective of this formula
	 * @return the connective of this formula
	 */
	public CType getConnective() {
		return connective;
	}
	
	
	/**
	 * add a new quantifier block to the quantifier prefix
	 * @param q the quantifier block to be added
	 */
	public void addQBlock(QBlock q) {
		quantifiers.addQuantBlock(q);
	}
	
/******************************************
 * Print the formula in qpro format (without header)
 ******************************************/
	
	
	/**
	 * prints the formula in qpro format
	 */
	
	public void print() {
		printFormula(this);
	}
	
	
	// print the formula
	
	private void printFormula(Formula f) {
		
		if (f.getQuantifiers().size() != 0) {
			printQuantifiers(f);
			return;
		}
		
		if (f.getConnective() == CType.AND) {
			printConjunction(f);
		} else {
			printDisjunction(f);
		}
	}
	
	// print disjunction
	
	private void printDisjunction(Formula f) {
		System.out.println("d");
		
		printConnective(f);
		
		System.out.println("/d");
		
	}

	
	// print conjunction
	
	private void printConjunction(Formula f) {
		System.out.println("c");
		
		printConnective(f);
		
		System.out.println("/c");
		
	}
	
	
	// print internals of a connective
	
	private void printConnective(Formula f) {
		
		for(VOccurrence v : f.getPosLits()) {				// print positive literals
			System.out.print(v.getVariable() + " ");
		}
		
		System.out.println();
		
		for(VOccurrence v : f.getNegLits()) {				// print negative literals
			System.out.print(v.getVariable() + " ");
		}
		System.out.println();
		
		for(Formula sf : f.getSubformulas()) {				// print subformulas
			printFormula(sf);
		}
	}


	// print quantifiers

	private void printQuantifiers(Formula f){
		Quantifiers q = f.getQuantifiers();
		
		System.out.println("q");
													// print quantifier prefix
		for (QBlock b : q.getQuantifierBlocks()) {
			if (b.getQType() == QType.EXISTS) {
				System.out.print("e ");
			} else {
				System.out.print("a ");
			}
			
			for (IVar v : b.getVars()) {
				System.out.print(v.getName()+" ");
			}
			
			System.out.println();
		}
		
													// print formula 
		
		if (f.getConnective() == CType.AND) {
			printConjunction(f);
		} else {
			printDisjunction(f);
		}
		
		System.out.println("/q");
		
	}
	
	public void printBoole() {
		if (quantifiers.size() != 0) {
			printQuantifiers();
			return;
		}
		
		printConnective();
	}
	
	private void printQuantifiers() {
		int count = 0;
		for(QBlock qb : quantifiers.getQuantifierBlocks()) {
			if (qb.getVars().size() != 0) {
				if (qb.getQType() == QType.FORALL) {
					System.out.print("forall [");
				} else {
					System.out.print("exists [");
				}
				
				for (IVar vi :qb.getVars()) {
					System.out.print("v"+vi.getName()+" ");
				}
				System.out.print("](");
				count++;
			}
			
		}
		printConnective();
		for (int i = 0; i < count; i++) {
			System.out.print(")");
		}
	}
	
	private void printConnective() {
		char token;
		
		if (connective == CType.AND) {
			token = '&';
		} else {
			token = '|';
		}
		
		int count = plits.size() + nlits.size()+ subFormulas.size();
		
		for (VOccurrence c : plits) {
			//if (c.getPolarity() == false) System.out.print("!");
			System.out.print("v"+c.getVariable());
			if (count > 1) {
				System.out.print(" " + token + " ");
			}
			count--;
		}
		
		for (VOccurrence c : nlits) {
			//if (c.getPolarity() == false) System.out.print("!");
			System.out.print("!v"+c.getVariable());
			if (count > 1) {
				System.out.print(" " + token + " ");
			}
			count--;
		}
		
		for (Formula f : subFormulas) {
			System.out.print("(");
			f.printBoole();
			System.out.print(")");
			if (count > 1) {
				System.out.print(" " + token + " ");
			}
			count--;
		}
	}

	public void setQuantifier(Quantifiers q) {
		quantifiers = q;
	}
	

}
