package at.jku.fmv.qbf.nnf.formula;

import java.util.HashMap;

/**
 * This class contains some global data about a formula:
 * varNumber:		the number of vars occurring in the formula (as specified in the formulas header)
 * vars:			infos about the variables
 * root:			the root of the formula treee
 * nameOfVars:		a mapping from the names of the vars in the serialization of the formula to the internal names
 */

public class FormulaData {
	private Variables vars;							// infos about the variables
	private int varNumber;							// number of variables occurring in the formula			
	private HashMap<Integer,Integer> nameOfVars;	// mapping orig. names to internal names
	
	private Formula root;							// the root node in the formula tree
	
	private boolean [] varInScope; 					// necessary for scope check during parsing
	
	
	/**
	 * inits the formula data of a formula
	 * @param varNumber number of variables occurring in the formula
	 * @param rootFormula root node of the formula
	 */
	public FormulaData(int varNumber, Formula rootFormula) {
		
		this.root = rootFormula;
		vars = new Variables();
		this.varNumber = varNumber;
		nameOfVars = new HashMap<Integer,Integer> ();			
		
		varInScope = new boolean[varNumber];
	}
	
	
	/**
	 * inits the formula data of a formula where the number of variables is unknown
	 * @param rootFormula root node of the formula
	 */
	public FormulaData(Formula rootFormula) {
		
		this.root = rootFormula;
		vars = new Variables();
		varNumber = Integer.MAX_VALUE;
		nameOfVars = new HashMap<Integer,Integer> ();
		
		varInScope = new boolean[varNumber];
	}
	
	/**
	 * set the number of vars occurring in the formula
	 * @param varNumber number of vars occurring in the formula
	 */
	public void setVarNumber(int varNumber) {
		this.varNumber = varNumber;
	}
	
	/**
	 * returns the number of variables occurring in the formula
	 * @return number of variables in the formula
	 */
	public int getVarNumber() {
		return varNumber;
	}
	
	/**
	 * sets a new root node for the formula
	 * @param root - the new root node
	 */
	public void setFormula(Formula root) {
		this.root = root;
	}
	
	/**
	 * returns the root node of the contained formula
	 * @return root node
	 */
	public Formula getFormula() {
		return root;
	}

	/**
	 * prints the formula according to the qdimacs format (incl. header)
	 */
	public void print() {
		System.out.println("QBF\n" + varNumber);
		root.print();
	}


	/**
	 * add a new positive occurrence of a variable to the formula
	 * @param v the variable occurrence to add
	 * @param f the subformula of root where the variable occurrence is inserted
	 * checks if var is in scope of a quantifier
	 */
	public void addPosVarOcc(int v, Formula f) {
		
		assert varInScope[v] : "variable not in scope of quantifier";
		
		assert v <= varNumber : "variable too big: " + v;
		assert nameOfVars.get(v) != 0 : "variable not quantified: " + v;
		
		VOccurrence vOcc = new VOccurrence(nameOfVars.get(v));
		vars.addPosOccurrence(vOcc);		
		f.addPlit(vOcc);
	}

	
	/**
	 * add a new negative occurrence of a variable to the formula
	 * @param v the variable occurrence to add
	 * @param f the subformula of root where the variable occurrence is inserted
	 * checks if var is in scope of a quantifier
	 */
	public void addNegVarOcc(int v, Formula f) {
		
		assert varInScope[v] : "variable not in scope of quantifier";
		
		assert v <= varNumber : "variable too big: " + v;
		assert nameOfVars.get(v) != 0 : "variable not quantified: " + v;
		
		VOccurrence vOcc = new VOccurrence(nameOfVars.get(v));
		vars.addNegOccurrence(vOcc);		
		f.addNlit(vOcc);	
	}

	/**
	 * adds a new variable to a given quantifier block
	 * @param var the new variable
	 * @param qblock the quantifier block where the new variable is inserted
	 * checks if the variable is not already quantified
	 */
	public void addVar(int var, QBlock qblock) {
		
		varInScope[var] = true;
		
		assert (nameOfVars.get(var) == null) : "variable is already quantified: " + var;
		
		VariableInfo vinfo = new VariableInfo(qblock);
		
		int vi = vars.addVar(vinfo);	
		nameOfVars.put(var,vi);
		qblock.add(vinfo);
	}
	
	public void removeVarFromScpe(int var) {
		varInScope[var] = false;
	}
	
	public void printBoole() {
		root.printBoole();
	}
	
}
