package at.jku.fmv.qbf.nnf.parser;


// assumption: 0, 1 represent truth constants


import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.LinkedList;
import java.util.List;

import at.jku.fmv.qbf.nnf.formula.CType;
import at.jku.fmv.qbf.nnf.formula.Formula;
import at.jku.fmv.qbf.nnf.formula.FormulaData;
import at.jku.fmv.qbf.nnf.formula.QBlock;
import at.jku.fmv.qbf.nnf.formula.QType;

/** 
 * parser for qpro format
 * 
 * @author Martina Seidl
 *
 */
public class Parser {
	private BufferedReader input;						// reading the input	
	private int varNumber;						// number of variables
	private FormulaData fd;						// data of the formula

	int q1 = 0, q2 = 0, c1 = 0, c2 = 0, d1 = 0, d2 = 0;
	
	/**
	 * init parser - read from input stream
	 * @param in input stream to read from
	 */
	private Parser(InputStream in){
		input = new BufferedReader(new InputStreamReader(in));
	}
	
	/**
	 * init parser - read from file
	 * @param in file to read from
	 */
	private Parser(File in) {
		try {
			input = new BufferedReader(new FileReader(in));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * parse QBF in qpro format 
	 * @param in stream to read from
	 * @return formula data
	 * @throws IOException 
	 * @throws ParseException 
	 */
	public static FormulaData parse(InputStream in) throws IOException, ParseException {
		Parser p = new Parser(in);
		p.startParsing(true);
		return p.getFormulaData();
	}
	
	public static FormulaData parse(InputStream in, boolean neg) throws IOException, ParseException {
		Parser p = new Parser(in);
		p.startParsing(neg);
		return p.getFormulaData();
	}
	
	
	/**
	 * parse QBF in qpro format 
	 * @param in file to read from
	 * @return formula data
	 * @throws IOException 
	 * @throws ParseException 
	 */
	public static FormulaData parse(File in) throws IOException, ParseException {
		Parser p = new Parser(in);
		p.startParsing(true);
		return p.getFormulaData();
	}

	public static FormulaData parse(File in,boolean neg) throws IOException, ParseException {
		Parser p = new Parser(in);
		p.startParsing(neg);
		return p.getFormulaData();
	}
	
	// get the formula data of the parsed formula
	
	private FormulaData getFormulaData() {
		return fd;
	}
	
	
	// start the parsing process
	
	
	
	
	private void startParsing(boolean neg) throws IOException, ParseException {
		Formula rootFormula;

		
		while(!input.readLine().trim().equals("QBF")); 				// skip comments
		
		varNumber = Integer.valueOf(input.readLine());						// set Number of Variables
		
		assert varNumber >= 1 : "invalid number of variables: " + varNumber;
		
		varNumber += 2;										// consider truth constants
		
		
		rootFormula = new Formula();						// init the root formula
		
		fd = new FormulaData(varNumber, rootFormula);		// init the formula data	
		
				
		String line = input.readLine();	// do parsing
		switch(line.charAt(0)) {
			case 'd': 	parseDisjunction(rootFormula, neg);
						break;
			case 'c':	parseConjunction(rootFormula, neg);
						break;
			case 'q':	parseQuantifier(rootFormula, neg);
						break;
			default:	throw new ParseException(line+": inexpected input");
		}			
	}
	
	
	// parse quantifiers
	
	private void parseQuantifier(Formula f, boolean pol) throws IOException, ParseException {
		q1++;
		
		LinkedList <Integer> l = new LinkedList<Integer>();
														// parse quantifier blocks
		String q = input.readLine();					// parse type of quantifier	
		char qtype = q.charAt(0);
		parseVars(q.substring(1), QType.convert(qtype,pol), f, l);	// parse variables in scope of quantifer
		
		
		while (true) {
			q = input.readLine();											// parse next element

								
			if (q.charAt(0) == qtype) {					// assert that quantifier blocks of successive blocks alternate
				throw new ParseException("quantifier blocks must be alternating");
			} // TODO alternation
			
			qtype = q.charAt(0);					
			switch(q.charAt(0)) {
				case 'd':				// parse disjunction
							parseDisjunction(f,pol);
							q = input.readLine();
							if (!q.trim().equals("/q")) {
								throw new ParseException("missing closing /q");
							}
							unsetScopes(l);
							
							return;
				case 'c':			// parse conjunction
							parseConjunction(f,pol);
							
							unsetScopes(l);
							q = input.readLine();
							
							if (!q.equals("/q")) {
								throw new ParseException("missing closing /q");
							}
							return;						// parse next quantifier block
				case 'q':	//input.nextLine();
							
							parseQuantifier(f,pol);
							q = input.readLine();
							
							if (!q.equals("/q")) {
								throw new ParseException("missing closing /q");
							}
							return;						// parse next quantifier 
							
			
				default:	parseVars(q.substring(1), QType.convert(qtype,pol), f, l);
			}
		}
		
		
	}
	
	private void unsetScopes(List <Integer> l) {
		for (Integer i : l) {
			fd.removeVarFromScpe(i);
		}
		
	}
	
	
	// parse variables in a quantifier block
	// vars: line containing variables
	// type: type of quantifier block
	// f: formula where quantifier block is attached to 
	
	private void parseVars(String line, QType type, Formula f, List <Integer>l) {
		QBlock qb = new QBlock(type, f);			// init qblock and add to formula f
	
		
		String [] vars = line.trim().split(" ");
		
		int var;
		
		if (line.length() != 0) {
		for(int i = 0; i < vars.length; i++) {
			var = new Integer(vars[i]);
			
			l.add(var);
			
			assert var >= 2 : "invalid value for var: " + var;
			assert var <= varNumber : "invalid value for var: " + var;
			
			fd.addVar(var, qb);							// add new variables
		}
		}
		
	}
	
	
	// parse variable occurrences
	// lits: list of variable occurrences
	// pol: polarity of the variable occurrences
	// f: direct superformula
	private void parseLits(String line, boolean pol, Formula f) {
	
		
		if (line.trim().length() == 0) return;
		
		String [] lits = line.trim().split(" "); // get variable occurrences
		
		int lit;
		
		for (int i = 0; i < lits.length; i++) {
			lit = Integer.valueOf(lits[i]);
			
			if (pol) {
				fd.addPosVarOcc(lit, f);				// add pos occurrence
			}
			else {
				fd.addNegVarOcc(lit, f);				// add neg occcurrence
			}
		}
		
		
	}

	
	// parse a disjunction
	
	private void parseDisjunction(Formula f, boolean pol) throws IOException, ParseException {
		if (pol) f.setConnective(CType.OR);	
		else f.setConnective(CType.AND);	
		
		String posLits = input.readLine();
		
		parseLits(posLits,pol,f);						// parse positive literals
		
		String negLits = input.readLine();			
		parseLits(negLits,!pol,f);						// parse negative literals
		
		String next;
		
		Formula tmp;
		
													// parse subformulas
		while(!(next = input.readLine()).equals("/d")) {
			
			tmp = new Formula();
			
			switch(next.charAt(0)) {
				case 'q':	//input.nextLine();
							parseQuantifier(tmp,pol);
							break;
				case 'c':	//input.nextLine();
							parseConjunction(tmp,pol);
							break;
				default :  	throw new ParseException("unexpected token");
			}
			
			
			f.addSubFormula(tmp);						// add subformulas
			
		}
		
	//	input.nextLine();
		
	}
	
	// parse a conjunction
	
	private void parseConjunction(Formula f, boolean pol) throws IOException, ParseException {
		if (pol) f.setConnective(CType.AND);
		else f.setConnective(CType.OR);
		
		String posLits = input.readLine();
		
		parseLits(posLits,pol,f);
		
		
		String negLits = input.readLine();
		
		parseLits(negLits,!pol,f);
		
		String next;
		Formula tmp;
		
		while(!(next = input.readLine()).equals("/c")) {
		//while(!(next = input.next("[qd]|/c")).equals("/c")) {
			
			tmp = new Formula();
			switch(next.charAt(0)) {
				case 'q':	//input.nextLine();
							parseQuantifier(tmp,pol);
							break;
				case 'd':	//input.nextLine();
							parseDisjunction(tmp,pol);
							break;
				default :	throw new ParseException("unexpected token");
			}
			f.addSubFormula(tmp);	
		}
		//input.nextLine();
	
	}
		
	

	
/*******************
 * main method for test purposes
 ********************/
	
	public static void main(String args[]) {
		//System.out.println("starting parsing");
		
		long zstVorher;
		long zstNachher;

		zstVorher = System.currentTimeMillis();

		// Aufruf lange dauernder Prozesse

		
		
		try {
			FormulaData fd = Parser.parse(new File("tests/poss9.qpro"));
			fd.print();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		
		zstNachher = System.currentTimeMillis();
		System.out.println("Zeit benötigt: " + ((zstNachher - zstVorher)) + " ms");
		
		
		
		zstNachher = System.currentTimeMillis();
		System.out.println("\nZeit benötigt: " + ((zstNachher - zstVorher)) + " ms");
	}
}
